import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import {
  OutletBranchDto,
  PaginationDto,
  PaginationSDto,
  StatusDto,
} from './dto/outlet-branch.dto';
import { OutletBranchService } from './outlet-branch.service';
import { AuthGuard } from '@nestjs/passport';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { UserRole, PermissionAction } from 'src/enum';

@Controller('outlet-branch')
export class OutletBranchController {
  constructor(private readonly outletBranchService: OutletBranchService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'outlet_branch'])
  create(@Body() createOutletBranchDto: OutletBranchDto) {
    return this.outletBranchService.create(createOutletBranchDto);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'outlet_branch'])
  findAll(@Query() dto: PaginationSDto) {
    return this.outletBranchService.findAll(dto);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'outlet_branch'])
  findOne(@Param('id') id: string) {
    return this.outletBranchService.findOne(id);
  }

  @Get('list/:companyId')
  find(@Query() dto: PaginationSDto, @Param('companyId') companyId: string) {
    return this.outletBranchService.find(dto, companyId);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'outlet_branch'])
  update(
    @Param('id') id: string,
    @Body() updateOutletBranchDto: OutletBranchDto,
  ) {
    return this.outletBranchService.update(id, updateOutletBranchDto);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'outlet_branch'])
  status(@Param('id') id: string, @Body() dto: StatusDto) {
    return this.outletBranchService.status(id, dto);
  }
}
