import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Post,
  Put,
  Patch,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { MenusService } from 'src/menus/menus.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { AccountService } from './account.service';
import { CreateAccountDto, PaginationDto, StatusDto, PasswordDto, PasswordWithOldDto } from './dto/account.dto';
import { Account } from './entities/account.entity';

@Controller('account')
export class AccountController {
  constructor(
    private readonly accountService: AccountService,
    private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
  ) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'account'])
  async create(@Body() dto: CreateAccountDto, @CurrentUser() user: Account) {
    dto.settingId = user.settingId;
    const account = await this.accountService.create(dto, user.id);
    const menus = await this.menuService.findAll();
    const perms = await this.permissionService.findAll();

    const obj = [];
    menus.forEach((menu) => {
      perms.forEach((perm) => {
        obj.push({
          accountId: account.id,
          menuId: menu.id,
          permissionId: perm.id,
        });
      });
    });
    this.userPermService.create(obj);
    return account;
  }

  @Get('my-staff')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'account'])
  find(@Query() query: PaginationDto, @CurrentUser() user: Account) {
    return this.accountService.find(query, user.id);
  }

  @Get('my-staff/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'account'])
  findById(@Query() query: PaginationDto, @Param('id') id: string) {
    return this.accountService.find(query, id);
  }

  @Get('outlet/profile')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.SUB_OUTLET)
  outletProfile(@CurrentUser() user: Account) {
    return this.accountService.outletProfile(user.id);
  }

  @Get('outlet/delivery-boy/profile')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.DELIVERY_BOY)
  profile(@CurrentUser() user: Account) {
    return this.accountService.deliveryProfile(user.id);
  }

  @Get('vendors')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'account'])
  findAll(@Query() query: PaginationDto) {
    return this.accountService.findAll(query);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'account'])
  findOne(@Param('id') id: string) {
    return this.accountService.findOne(id);
  }

  @Patch('password')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'account'])
  updateOwnPassword(
    @Body() dto: PasswordWithOldDto,
    @CurrentUser() user: Account,
  ) {
    return this.accountService.updateOwnPassword(dto, user.id);
  }

  @Patch('password/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  updatePassword(@Body() dto: PasswordDto, @Param('id') id: string) {
    return this.accountService.updatePassword(dto, id);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'account'])
  status(@Param('id') id: string, @Body() dto: StatusDto) {
    dto.wrongCount = 0;
    return this.accountService.status(id, dto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.DELETE, 'account'])
  remove(@Param('id') id: string) {
    return this.accountService.remove(id);
  }
}
