import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { CityDto } from './dto/city.dto';
import { City } from './entities/city.entity';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';

@Injectable()
export class CityService {
  constructor(
    @InjectRepository(City)
    private readonly repo: Repository<City>,
  ) {}

  async create(dto: CityDto) {
    const category = await this.repo.findOne({
      where: { name: dto.name, stateId: dto.stateId },
    });
    if (category) {
      throw new ConflictException('City already exists!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(
    limit: number,
    offset: number,
    keyword: string,
    status: boolean,
    stateId: number,
  ) {
    const [result, total] = await this.repo
      .createQueryBuilder('city')
      .where('city.status = :status AND city.stateId = :stateId', {
        status: status,
        stateId: stateId,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('city.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN city.name LIKE '${keyword}%' THEN 0 ELSE 1 END, city.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();

    return { result, total };
  }

  async find(limit: number, offset: number, keyword: string, stateId: number) {
    const [result, total] = await this.repo
      .createQueryBuilder('city')
      .where('city.status = :status AND city.stateId = :stateId', {
        status: true,
        stateId: stateId,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('city.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN city.name LIKE '${keyword}%' THEN 0 ELSE 1 END, city.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: number) {
    const category = await this.repo.findOne({ where: { id } });
    if (!category) {
      throw new NotFoundException('City not found!');
    }
    return category;
  }

  async update(id: number, dto: CityDto) {
    try {
      const category = await this.repo.findOne({ where: { id } });
      if (!category) {
        throw new NotFoundException('City not found!');
      }
      const obj = Object.assign(category, { name: dto.name });
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either catgeory exists or invalid name!',
      );
    }
  }

  async status(id: number, dto: BoolStatusDto) {
    const menu = await this.repo.findOne({ where: { id } });
    if (!menu) {
      throw new NotFoundException('City not found!');
    }
    const obj = Object.assign(menu, dto);
    return this.repo.save(obj);
  }
}
