// eslint-disable-next-line prettier/prettier
import { ApiProperty } from '@nestjs/swagger';
import { Type } from 'class-transformer';
import {
  IsArray,
  IsDate,
  IsEnum,
  IsNotEmpty,
  IsNumber,
  IsOptional,
  IsString,
  IsUUID,
  Max,
  MaxLength,
  Min,
  MinLength,
} from 'class-validator';
import { OrderStatus } from 'src/enum';

export class OrderDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsUUID()
  subCategoryId: string;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  quantity: number;

  @IsOptional()
  @IsUUID()
  outletDetailId: string;

  @IsOptional()
  cartId: string;
}

export class UpdateOrderDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsNotEmpty()
  id: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsUUID()
  subCategoryId: string;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(1)
  quantity: number;
  
  @IsOptional()
  remark: string;

  @IsOptional()
  cartId: string;

  @IsOptional()
  status: OrderStatus;

  @IsOptional()
  date: Date;
  
  @IsOptional()
  remarkStatus: boolean;
  
  @IsOptional()
  createdAt: Date;
}

export class OrderJsonDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsArray()
  data: UpdateOrderDto[];

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Date)
  @IsDate()
  date: Date;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Boolean)
  status: boolean;
}

export class StatusOrderDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsNotEmpty()
  id: string;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  quantity: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  delivered: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  returned: number;

  @ApiProperty()
  @IsOptional()
  @MinLength(0)
  @MaxLength(255)
  reason: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(OrderStatus)
  status: OrderStatus;
}

export class JsonOrderDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsArray()
  data: StatusOrderDto[];
}

export class UpdateOrderQtyDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsUUID()
  id: string;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  delivered: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  returned: number;

  @ApiProperty()
  @IsNotEmpty()
  @MinLength(0)
  @MaxLength(255)
  reason: string;
}

export class StatusDto {
  @ApiProperty()
  @IsOptional()
  cartId: string;
}

export class RemarkDto {
  @ApiProperty()
  @IsNotEmpty()
  @MinLength(30)
  @MaxLength(3000)
  remark: string;
}

export class PaginationDto {
  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(100)
  limit: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  offset: number;

  @ApiProperty()
  @IsOptional()
  @IsString()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @ApiProperty()
  @IsOptional()
  fromDate: Date;

  @ApiProperty()
  @IsOptional()
  toDate: Date;
}

export class SearchListDto {
  @ApiProperty()
  @IsOptional()
  @IsString()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;
}

export class DeliveryBoyDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsUUID()
  accountId: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(OrderStatus)
  status: OrderStatus;
}

export class PaginationAllDto {
  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(100)
  limit: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  offset: number;

  @ApiProperty()
  @IsOptional()
  @IsString()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsNotEmpty()
  @ApiProperty()
  @IsEnum(OrderStatus)
  status: OrderStatus;

  @ApiProperty()
  @IsNotEmpty()
  fromDate: Date;

  @ApiProperty()
  @IsNotEmpty()
  toDate: Date;
}

export class PaginationAllReportDto {
  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(100)
  limit: number;

  @ApiProperty()
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  offset: number;

  @ApiProperty()
  @IsOptional()
  @IsString()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsNotEmpty()
  @ApiProperty()
  @IsEnum(OrderStatus)
  status: OrderStatus;

  @ApiProperty()
  @IsNotEmpty()
  fromDate: Date;

  @ApiProperty()
  @IsNotEmpty()
  toDate: Date;

  @ApiProperty()
  @IsString()
  @MinLength(0)
  @MaxLength(100)
  categoryName: string;

  @ApiProperty()
  @IsOptional()
  allorder: string;
}

export class PaginationReportDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(50)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  @Max(100)
  offset: number;

  @ApiProperty()
  @IsOptional()
  fromDate: Date;

  @ApiProperty()
  @IsOptional()
  toDate: Date;
}
