import {
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { OrderStatus } from 'src/enum';
import { NotifyService } from 'src/notify/notify.service';
import { OutletDetailsService } from 'src/outlet-details/outlet-details.service';
import { Brackets, Repository } from 'typeorm';
import {
  OrderDto,
  PaginationAllDto,
  PaginationAllReportDto,
  PaginationReportDto,
  PaginationDto,
  RemarkDto,
  SearchListDto,
  StatusDto,
  StatusOrderDto,
  UpdateOrderDto,
} from './dto/order.dto';
import { Order } from './entities/order.entity';

@Injectable()
export class OrdersService {
  constructor(
    @InjectRepository(Order) private readonly repo: Repository<Order>,
    private readonly outletDetailService: OutletDetailsService,
    private readonly notifyService: NotifyService,
  ) {}

  async create(dto: OrderDto, outletDetailId: string) {
    const query = this.repo
      .createQueryBuilder('order')
      .where(
        'order.outletDetailId = :outletDetailId AND order.subCategoryId = :subCategoryId AND order.status IN (:...status)',
        {
          outletDetailId,
          subCategoryId: dto.subCategoryId,
          status: [OrderStatus.CART, OrderStatus.ORDERED],
        },
      );
    if (dto.cartId && dto.cartId.length > 3 && dto.cartId != 'null') {
      query.andWhere('order.cartId = :cartId', {
        cartId: dto.cartId,
      });
    } else {
      query.andWhere('order.cartId IS NULL');
    }
    const result = await query.getOne();

    if (result) {
      const obj = Object.assign(result, { quantity: dto.quantity });
      return this.repo.save(obj);
    }
    if (dto.cartId == 'null') {
      dto.cartId = null;
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async update(date: any, dto: UpdateOrderDto[], id: string, fcm: string) {
    let cartId = null;
    const curday = new Date();
    const activeObjects = dto.filter(
      (item) => item.status === OrderStatus.ORDERED,
    );
    if (activeObjects.length > 0) {
      cartId = activeObjects[0].cartId;
    } else {
      const data = await this.outletDetailService.findOrderId(id);
      const storeName = await this.findFirstWords(data.storeName);
      const today = new Date();
      const day = await today.getDate().toString().padStart(2, '0'); // Ensure two-digit format
      const areaName = await this.findFirstWords(data.area['name']);
      const randomNumber =
        (await Math.floor(Math.random() * (9999 - 1000 + 1))) + 1000;

      cartId = storeName + day + areaName + randomNumber;
    }

    const hasRemarks = dto.some(item => item.remark.trim() !== ''); //check total order is any remark

    dto.forEach((element) => {
      element.cartId = cartId.toString();
      element.status = OrderStatus.ORDERED;
      element.date = date;
      element.remarkStatus = hasRemarks;
      element.createdAt = curday;
    });
    console.log(dto)
    this.repo.save(dto);
    this.notifyService.orderPlaced(fcm, cartId, date, id);
    return { result: dto };
  }

  private findFirstWords(a: string) {
    return a.match(/\b\w/g).join('').toUpperCase();
  }

  async updateQty(id: string, dto: StatusOrderDto) {
    const order = await this.repo.findOne({ where: { id } });
    if (!order) {
      throw new NotFoundException('Order not found!');
    }
    const obj = Object.assign(order, dto);
    return this.repo.save(obj);
  }

  async remark(id: string, dto: RemarkDto) {
    const order = await this.repo.findOne({ where: { id } });
    if (!order) {
      throw new NotFoundException('Order not found!');
    }
    const obj = Object.assign(order, dto);
    const result = await this.repo.save(obj);
    return { result };
  }

  async updateDeliveryBoy(cartId: string, accountId: string, status: any) {
    try {
      return this.repo
        .createQueryBuilder()
        .update()
        .set({
          status: status,
          accountId: accountId,
        })
        .where('cartId = :cartId', { cartId: cartId })
        .execute();
    } catch (error) {
      throw new NotAcceptableException('Invalid delivery boy id!');
    }
  }

  async signature(cartId: string, image: string) {
    try {
      this.repo
        .createQueryBuilder()
        .update()
        .set({
          signature: process.env.CDN_LINK + image,
          signaturePath: image,
        })
        .where('cartId = :cartId', { cartId: cartId })
        .execute();
      return { url: process.env.CDN_LINK + image };
    } catch (error) {
      throw new NotAcceptableException('Try after some time!');
    }
  }

  async returnProduct(id: string, image: string) {
    try {
      this.repo
        .createQueryBuilder()
        .update()
        .set({
          returnImage: process.env.CDN_LINK + image,
          returnImagePath: image,
        })
        .where('id = :id', { id: id })
        .execute();
      return { url: process.env.CDN_LINK + image };
    } catch (error) {
      throw new NotAcceptableException('Try after some time!');
    }
  }

  status(dto: StatusOrderDto[]) {
    dto.forEach((element) => {
      if (element.delivered == 0) {
        element.status = OrderStatus.RETURNED;
      }
      if (element.returned > 0) {
        element.status = OrderStatus.PARTIALLY_DELIVERED;
      }
      if (element.returned == 0) {
        element.status = OrderStatus.DELIVERED;
      }
    });
    this.repo.save(dto);
    return { result: dto };
  }

  async outletCartList(outletDetailId: string, dto: StatusDto) {
    const query = this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.subCategory', 'subCategory')
      .leftJoinAndSelect('subCategory.category', 'category')
      .select([
        'order.id',
        'order.quantity',
        'order.returned',
        'order.delivered',
        'order.cartId',
        'order.status',
        'order.date',
        'order.reason',
        'order.remark',
        'order.createdAt',

        'subCategory.id',
        'subCategory.name',
        'subCategory.image',

        'category.id',
        'category.name',
      ])
      .where('order.outletDetailId = :outletDetailId', {
        outletDetailId: outletDetailId,
      });
    if (dto.cartId && dto.cartId.length > 3 && dto.cartId != 'null') {
      query.andWhere('order.cartId = :cartId', {
        cartId: dto.cartId,
      });
    } else {
      query.andWhere('order.cartId IS NULL');
    }
    const result = await query
      .andWhere('order.status IN (:...status1)', {
        status1: [OrderStatus.CART, OrderStatus.ORDERED],
      })
      .orderBy({ 'order.createdAt': 'DESC' })
      .getRawMany();

    return { result, total: result.length };
  }

  async adminOrderList(dto: PaginationAllDto) {
    const keyword = dto.keyword || '';
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const query = this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletDetail.routeMaster', 'routeMaster')
      .leftJoinAndSelect('routeMaster.account', 'account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .select([
        'outletDetail.id',
        'outletDetail.storeName',
        'outletDetail.address',

        'routeMaster.id',
        'routeMaster.routeName',

        'account.id',
        'staffDetail.name',

        'outletBranch.id',
        'outletBranch.branchName',

        'companyDetail.id',
        'companyDetail.businessName',

        'order.id',
        'order.cartId',
        'order.status',
        'order.date',
        'order.createdAt',
        'order.remarkStatus'
      ]);
      if (keyword.length > 0) {
        query.andWhere(
          new Brackets((qb) => {
            qb.where('companyDetail.businessName LIKE :cid and order.date >= :fromDate and order.date <= :toDate', {
              cid: '%' + keyword + '%',
              fromDate: fromDate,
              toDate: toDate,
            });
          }),
        );
      } else {
        query.where(
          'order.date >= :fromDate and order.date <= :toDate',
          { fromDate: fromDate, toDate: toDate },
        );
      }
      if (dto.status) {
        query.andWhere('order.status = :status', {
          status: dto.status,
        });
      }
    const [result, total] = await query
      .orderBy({ 'order.createdAt': 'DESC' })
      .groupBy('order.cartId')
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();
    return { result, total };
  }

  async adminOrderReport(dto: PaginationAllReportDto) {
    const keyword = dto.keyword || '';
    const fromDate = dto.fromDate;
    const toDate = dto.toDate;
    const categoryName = dto.categoryName;
    const cartId = JSON.parse(dto.allorder);

    const query = this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.subCategory', 'subCategory')
      .leftJoinAndSelect('subCategory.category', 'category')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletDetail.routeMaster', 'routeMaster')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .select([
        'outletDetail.id',
        'outletDetail.storeName',
        'outletDetail.address',

        'order.id',
        'order.cartId',
        'order.status',
        'order.date',
        'order.quantity',
        'order.delivered',
        'order.returned',
        'order.reason',
        'order.remark',
        'order.createdAt',

        'subCategory.id',
        'subCategory.name',
        'subCategory.image',
        'subCategory.foodType',

        'category.id',
        'category.name',

        'outletBranch.id',
        'outletBranch.branchName',

        'companyDetail.id',
        'companyDetail.businessName',

        "routeMaster.id",
        "routeMaster.routeName",
        "routeMaster.startPoint",
        "routeMaster.endPoint",
      ]);
      query.where(
        'order.date >= :fromDate and order.date <= :toDate and order.status = :status',
        { fromDate: fromDate, toDate: toDate, status: 'ORDERED'},
      );
      if (categoryName !== 'All') {
        query.andWhere('category.name = :categoryName', { categoryName: categoryName });
      }
      if (keyword.length > 0) {
        query.andWhere(
          new Brackets((qb) => {
            qb.where('companyDetail.businessName LIKE :cid', {
              cid: '%' + keyword + '%',
            });
          }),
        );
      }
      if (cartId.length > 0) {
        query.andWhere('order.cartId IN (:...cartId)', {
          cartId: [cartId],
        });
      }
    const result = await query
      .orderBy({ 'order.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getRawMany();
    return { result };
  }

    // this is for report section
  async findReportAll(dto: PaginationReportDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const query = this.repo
        .createQueryBuilder('order')
        .leftJoinAndSelect('order.outletDetail', 'outletDetail')
        .select([
            'outletDetail.id',
            'outletDetail.storeName',

            'order.id',
            'order.outletDetailId',
            'SUM(order.quantity) AS totalQuantity',
            'SUM(order.delivered) AS totalDelivered',
            'SUM(order.returned) AS totalReturned',
            'order.createdAt',
        ]);

    query.where(
        'order.createdAt >= :fromDate and order.createdAt <= :toDate',
        { fromDate: fromDate, toDate: toDate },
    );

    const result = await query
        .orderBy({ 'order.createdAt': 'DESC' })
        .groupBy('order.outletDetailId')
        .take(dto.limit)
        .skip(dto.offset)
        .getRawMany();

      const total = await query.getCount();
      return { result, total };
    }

  async outletOrderList(outletDetailId: string, dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const query = this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.subCategory', 'subCategory')
      .leftJoinAndSelect('subCategory.category', 'category')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .select([
        'outletDetail.id',
        'outletDetail.storeName',
        'outletDetail.address',

        'order.id',
        'order.cartId',
        'order.status',
        'order.tistatus',
        'order.date',
        'order.createdAt',
      ])
      .where(
        'order.outletDetailId = :outletDetailId AND order.status != :status',
        {
          outletDetailId: outletDetailId,
          status: OrderStatus.CART,
        },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where('order.id LIKE :oid OR order.cartId LIKE :cid', {
            oid: '%' + keyword + '%',
            cid: '%' + keyword + '%',
          });
        }),
      );
    if (dto.fromDate && dto.toDate) {
      query.andWhere('order.date >= :fromDate and order.date <= :toDate', {
        fromDate: fromDate,
        toDate: toDate,
      });
    }
    const [result, total] = await query
      .orderBy({ 'order.createdAt': 'DESC' })
      .groupBy('order.cartId')
      .getManyAndCount();
    return { result, total };
  }

  async outletOrderItems(cartId: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.subCategory', 'subCategory')
      .leftJoinAndSelect('subCategory.category', 'category')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .select([
        'outletDetail.id',
        'outletDetail.storeName',
        'outletDetail.address',
        'order.id',
        'order.quantity',
        'order.returned',
        'order.delivered',
        'order.cartId',
        'order.status',
        'order.date',
        'order.reason',
        'order.remark',
        'order.createdAt',

        'subCategory.id',
        'subCategory.name',
        'subCategory.image',

        'category.id',
        'category.name',
      ])
      .where('order.cartId = :cartId AND order.status IN (:...status)', {
        cartId: cartId,
        status: [
          OrderStatus.ORDERED,
          OrderStatus.DISPATCHED,
          OrderStatus.DELIVERED,
          OrderStatus.PARTIALLY_DELIVERED,
          OrderStatus.RETURNED,
          OrderStatus.DELETED,
        ],
      })
      .orderBy({ 'order.createdAt': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async printOrderItems(cartId: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.subCategory', 'subCategory')
      .leftJoinAndSelect('subCategory.category', 'category')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .select([
        'outletDetail.id',
        'outletDetail.outletId',
        'outletDetail.storeName',
        'outletDetail.address',
        'order.id',
        'order.quantity',
        'order.returned',
        'order.delivered',
        'order.cartId',
        'order.status',
        'order.date',
        'order.reason',
        'order.remark',
        'order.createdAt',

        'subCategory.id',
        'subCategory.name',
        'subCategory.image',

        'category.id',
        'category.name',
      ])
      .where('order.cartId = :cartId', {
        cartId: cartId,
      })
      .orderBy({ 'order.createdAt': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async findGroupByDelivery(accountId: string, dto: SearchListDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .select([
        'outletDetail.id',
        'outletDetail.outletId',
        'outletDetail.storeName',
        'outletDetail.contactPerson',
        'outletDetail.address',
        'outletDetail.orderOpenTime',
        'outletDetail.orderCloseTime',
        'outletDetail.contact',
        'outletDetail.latitude',
        'outletDetail.longitude',

        'outletBranch.id',
        'outletBranch.phone',
        'outletBranch.branchName',
        'outletBranch.logo',

        'order.id',
        'order.cartId',
        'order.date',
        'order.status',
        'order.createdAt',
      ])
      .where('order.accountId = :accountId AND order.status IN (:...status)', {
        accountId: accountId,
        status: [OrderStatus.DISPATCHED],
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('order.id LIKE :oid OR order.cartId LIKE :cid', {
            oid: '%' + keyword + '%',
            cid: '%' + keyword + '%',
          });
        }),
      )
      .orderBy({ 'order.createdAt': 'DESC' })
      .groupBy('order.cartId')
      .getManyAndCount();
    return { result, total };
  }

  async findGroupDeliveryHistory(accountId: string, dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const query = this.repo
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.outletDetail', 'outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .select([
        'outletDetail.id',
        'outletDetail.outletId',
        'outletDetail.storeName',
        'outletDetail.contactPerson',
        'outletDetail.address',
        'outletDetail.orderOpenTime',
        'outletDetail.orderCloseTime',
        'outletDetail.contact',
        'outletDetail.latitude',
        'outletDetail.longitude',

        'outletBranch.id',
        'outletBranch.branchName',

        'order.id',
        'order.cartId',
        'order.status',
        'order.date',
        'order.createdAt',
      ])
      .where('order.accountId = :accountId AND order.status IN (:...status)', {
        accountId: accountId,
        status: [
          OrderStatus.DELIVERED,
          OrderStatus.PARTIALLY_DELIVERED,
          OrderStatus.RETURNED,
        ],
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('order.id LIKE :oid OR order.cartId LIKE :cid', {
            oid: '%' + keyword + '%',
            cid: '%' + keyword + '%',
          });
        }),
      );
      if(dto.fromDate && dto.toDate){
      query.andWhere('order.date >= :fromDate and order.date <= :toDate', {
        fromDate: fromDate,
        toDate: toDate,
      });
    }
    const [result, total] = await query
      .orderBy({ 'order.createdAt': 'DESC' })
      .groupBy('order.cartId')
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: string) {
    const order = await this.repo.findOne({ where: { id } });
    if (!order) {
      throw new NotFoundException('Order not found!');
    }
    return order;
  }

  async removeorders(id: string) {
    const resultdata = await this.repo.find({ where: { cartId: id } });

    resultdata.forEach(async(element) => {
      const obj = Object.assign(element, {status: OrderStatus.REJECTED});
      this.repo.save(obj);
    });
  }

  async removesingle(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Not found!');
    }
    const obj = Object.assign(result, {status: OrderStatus.DELETED});
    return this.repo.save(obj);
  }
}
