import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  UseGuards,
  Delete,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { PermissionAction, UserRole } from 'src/enum';
import { PaginationSDto, RouteMasterDto, DeliveryBoyDto } from './dto/route-master.dto';
import { RouteMasterService } from './route-master.service';

@Controller('route-master')
export class RouteMasterController {
  constructor(private readonly RouteMasterService: RouteMasterService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'route_master'])
  create(@Body() dto: RouteMasterDto) {
    return this.RouteMasterService.create(dto);
  }

  @Get('list/all')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'route_master'])
  findAll(@Query() query: PaginationSDto) {
    const keyword = query.keyword || '';
    return this.RouteMasterService.findAll(
      query.limit,
      query.offset,
      keyword,
      query.status,
    );
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'route_master'])
  update(@Param('id') id: number, @Body() dto: RouteMasterDto) {
    return this.RouteMasterService.update(id, dto);
  }

  @Patch('assign-route-delivery-boy/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  assignRouteDeliveryBoy(
    @Param('id') id: number,
    @Body() dto: DeliveryBoyDto,
  ) {
    return this.RouteMasterService.updateDeliveryBoy(id, dto);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'route_master'])
  status(@Param('id') id: number, @Body() dto: BoolStatusDto) {
    return this.RouteMasterService.status(id, dto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'route_master'])
  remove(@Param('id') id: number) {
    return this.RouteMasterService.remove(id);
  }

  
}
