import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotAcceptableException,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { DefaultStatus, UserRole } from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import {
  AccDeleteReasonDto,
  CreateAccountDto,
  MemberPaginationDto,
  UpdateStaffDto,
  UpdateStaffPasswordDto,
} from './dto/account.dto';
import { Account } from './entities/account.entity';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import { Cache } from 'cache-manager';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import * as bcrypt from 'bcrypt';
import { StaffDetail } from 'src/staff-details/entities/staff-detail.entity';
import { DefaultStatusPaginationDto } from 'src/common/dto/default-status-pagination.dto';
import { Menu } from 'src/menus/entities/menu.entity';
import { MenusService } from 'src/menus/menus.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';

@Injectable()
export class AccountService {
  constructor(
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    @InjectRepository(StaffDetail)
    private readonly staffRepo: Repository<StaffDetail>,
    @InjectRepository(Menu)
    private readonly menuRepo: Repository<Menu>,
    private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
    private readonly nodemailerService: NodeMailerService,
  ) {}

  async create(dto: CreateAccountDto, businessAdminId: string) {
    const user = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.STAFF },
    });
    if (user) {
      throw new ConflictException('Email already exists!');
    }

    const encryptedPassword = await bcrypt.hash(dto.password, 13);
    const obj = Object.assign({
      email: dto.email,
      password: encryptedPassword,
      businessAdminId: businessAdminId,
      roles: UserRole.STAFF,
      status: dto.status,
    });
    const payload = await this.repo.save(obj);
    const object = Object.assign({
      staffId: '#' + Math.floor(1000 + Math.random() * 9000),
      salutation: dto.salutation,
      gender: dto.gender,
      fName: dto.fName,
      lName: dto.lName,
      mobile: dto.mobile,
      staffRole: dto.staffRole,
      accountId: payload.id,
    });
    await this.staffRepo.save(object);
    return payload;
  }

  async userList(dto: MemberPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.city', 'city')
      .leftJoinAndSelect('userDetail.avatar', 'avatar')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.totalCoin',
        'account.referralCode',
        'account.referredBy',
        'account.createdAt',

        'userDetail.id',
        'userDetail.email',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',

        'city.id',
        'city.name',
        'city.image',

        'avatar.id',
        'avatar.avatar',
        'avatar.avatarName',
        'avatar.avatarGender',
      ])
      .where('account.roles = :roles AND account.status = :status', {
        roles: UserRole.USER,
        status: dto.status,
      });
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR userDetail.name LIKE :keyword OR userDetail.gender LIKE :keyword OR userDetail.userName LIKE :keyword ',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'account.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async getStaffDetails(
    dto: DefaultStatusPaginationDto,
    businessAdminId: string,
  ) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.businessAdminId',
        'account.email',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdAt',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where(
        'account.roles = :roles AND account.status = :status AND account.businessAdminId = :businessAdminId',
        {
          roles: UserRole.STAFF,
          status: dto.status,
          businessAdminId: businessAdminId,
        },
      );

    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR staffDetail.fName LIKE :keyword OR staffDetail.lName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'account.createdAt': 'DESC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async getStaffProfile(accountId: string) {
    const staff = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.email',
        'account.businessAdminId',
        'account.roles',
        'account.status',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();

    const perms = await this.menuRepo
      .createQueryBuilder('menu')
      .leftJoinAndSelect('menu.userPermission', 'userPermission')
      .leftJoinAndSelect('userPermission.permission', 'permission')
      .where('userPermission.accountId = :accountId', {
        accountId: accountId,
      })
      .orderBy({ 'menu.title': 'ASC', 'permission.id': 'ASC' })
      .getMany();

    return { staff: staff, perms };
  }

  async userProfile(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.city', 'city')
      .leftJoinAndSelect('userDetail.avatar', 'avatar')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.totalCoin',
        'account.roles',
        'account.status',
        'account.reason',
        'account.referralCode',
        'account.referredBy',
        'account.createdAt',

        'userDetail.id',
        'userDetail.email',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',

        'city.id',
        'city.name',
        'city.image',

        'avatar.id',
        'avatar.avatar',
        'avatar.avatarName',
        'avatar.avatarGender',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!result) {
      throw new NotFoundException('User not found!');
    }
    return result;
  }

  async updateStaff(accountId: string, dto: UpdateStaffDto) {
    const result = await this.staffRepo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, dto);
    return this.staffRepo.save(obj);
  }

  async updateStaffPassword(accountId: string, dto: UpdateStaffPasswordDto) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    if (dto.loginId && dto.loginId.length > 0) {
      const obj = Object.assign(result, { phoneNumber: dto.loginId });
      await this.repo.save(obj);
    }
    if (dto.password && dto.password.length > 0) {
      const password = await bcrypt.hash(dto.password, 10);
      const obj = Object.assign(result, { password: password });
      await this.repo.save(obj);
    }
    return result;
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    if (dto.status == DefaultStatus.ACTIVE) {
      result.reason = null;
      const obj = Object.assign(result, dto);
      return this.repo.save(obj);
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async statusByUser(id: string, dto: AccDeleteReasonDto) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Feedback not found!');
      }
      const obj = Object.assign(result, {
        reason: dto.reason,
        status: DefaultStatus.DEACTIVE,
      });
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async deductCoin(coin: number, accountId: string) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    const obj = Object.assign(result, { totalCoin: result.totalCoin - coin });
    return this.repo.save(obj);
  }
}
