import { HttpService } from '@nestjs/axios';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { Cache } from 'cache-manager';
import { Account } from 'src/account/entities/account.entity';
import { DefaultStatus, LogType, LoginType, UserRole } from 'src/enum';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';
import APIFeatures from 'src/utils/apiFeatures.utils';
import { Repository } from 'typeorm';
import {
  AdminSigninDto,
  BusinessCreateDto,
  BusinessForgotDto,
  BusinessLoginDto,
  BusinessResetDto,
  EmailDto,
  ForgotPassDto,
  MemberSigninDto,
  OtpDto,
  SigninDto,
  StaffLoginDto,
  VerifyOtpDto,
} from './dto/login.dto';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { LoginHistory } from 'src/login-history/entities/login-history.entity';
import { sendOtp } from 'src/utils/sms.utils';

@Injectable()
export class AuthService {
  constructor(
    private jwtService: JwtService,
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserPermission)
    private readonly upRepo: Repository<UserPermission>,
    @InjectRepository(UserDetail)
    private readonly userDetailRepo: Repository<UserDetail>,
    @InjectRepository(LoginHistory)
    private readonly logRepo: Repository<LoginHistory>,
    private readonly httpService: HttpService,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    private readonly nodeMailerService: NodeMailerService,
  ) {}

  async adminLogin(loginId: string, password: string) {
    const admin = await this.getUserDetails(loginId, UserRole.ADMIN);
    const comparePassword = await bcrypt.compare(password, admin.password);
    if (!comparePassword) {
      throw new UnauthorizedException('Invalid Credentials');
    }
    const token = await APIFeatures.assignJwtToken(admin.id, this.jwtService);
    return { token };
  }

  async userLogin(dto: MemberSigninDto) {
    let user = await this.repo.findOne({
      where: { phoneNumber: dto.loginId, roles: UserRole.USER },
    });    
    if (!user) {
      const obj = Object.create({
        phoneNumber: dto.loginId,
        // type: LoginType.PHONE,
        roles: UserRole.USER,
        totalCoin: 100,
        // fcm,
      });
      user = await this.repo.save(obj);
      const udObj = Object.create({
        accountId: user.id,
      });
      await this.userDetailRepo.save(udObj);
    }
    if (user.status != DefaultStatus.ACTIVE) {
      throw new BadRequestException(
        'Your account is not active! Please contact to admin.',
      );
    }
    // const fcmObj = Object.assign(user,{fcm: fcm});
    // await this.repo.save(fcmObj);

    const otp = 783200;
    // const otp = Math.floor(100000 + Math.random() * 900000);
    // sendOtp(dto.loginId, otp, smsConfig);
    this.cacheManager.set(dto.loginId, otp, 10 * 60 * 1000);
    return { loginId: dto.loginId };
  }

  async userVerifyOtp(dto: OtpDto) {
    const user = await this.getUserDetails(dto.loginId, UserRole.USER);
    const sentOtp = await this.cacheManager.get(dto.loginId);

    // if (phoneNumber != '8092326469') {
    if (dto.otp != sentOtp) {
      throw new UnauthorizedException('Invalid otp!');
    }
    // }
    const token = await APIFeatures.assignJwtToken(user.id, this.jwtService);

    const check = user.userDetail[0].name;
    if (check == null) {
      return { token, latest: true };
    } else {
      return { token, latest: false };
    }
  }

  validate(id: string) {
    return this.getUserDetails(id);
  }

  findPermission(accountId: string) {
    return this.getPermissions(accountId);
  }

  private getPermissions = async (accountId: string): Promise<any> => {
    let result = await this.cacheManager.get('userPermission' + accountId);
    if (!result) {
      result = await this.upRepo.find({
        relations: ['permission', 'menu'],
        where: { accountId, status: true },
      });
      this.cacheManager.set(
        'userPermission' + accountId,
        result,
        7 * 24 * 60 * 60 * 1000,
      );
    }
    return result;
  };

  private getUserDetails = async (
    id: string,
    role?: UserRole,
  ): Promise<any> => {
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail');
    if (!role && role == UserRole.USER) {
      query.where('account.roles = :roles', { roles: UserRole.USER });
    }
    // if (!role && role == UserRole.BUSINESS) {
    //   query.where('account.roles IN (:...roles)', {
    //     roles: [UserRole.BUSINESS],
    //   });
    // }
    if (!role && role == UserRole.ADMIN) {
      query.where('account.roles IN (:...roles)', {
        roles: [UserRole.ADMIN, UserRole.STAFF],
      });
    }
    const result = await query
      .andWhere('account.id = :id OR account.phoneNumber = :phoneNumber', {
        id: id,
        phoneNumber: id,
      })
      .getOne();
    if (!result) {
      throw new UnauthorizedException('Account not found!');
    }
    return result;
  };
}
