import { Injectable, NotFoundException } from '@nestjs/common';
import { AvatarPaginationDto, CreateAvatarDto } from './dto/create-avatar.dto';
import { UpdateAvatarDto } from './dto/update-avatar.dto';
import { Avatar } from './entities/avatar.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { DefaultStatusPaginationDto } from 'src/common/dto/default-status-pagination.dto';
import { DefaultStatus } from 'src/enum';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';

@Injectable()
export class AvatarService {
  constructor(
    @InjectRepository(Avatar)
    private readonly repo: Repository<Avatar>,
  ) {}

  async create(image: string, dto: CreateAvatarDto) {
    const obj = Object.assign({
      avatar: process.env.CLU_CDN_LINK + image,
      avatarPath: image,
      avatarName: dto.avatarName,
      avatarGender: dto.avatarGender,
    });
    return this.repo.save(obj);
  }

  async findAll(dto: AvatarPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('avatar');
    if (dto.status && dto.status.length > 0) {
      query.andWhere('avatar.status = :status', {
        status: dto.status,
      });
    }
    if (dto.avatarGender && dto.avatarGender.length > 0) {
      query.andWhere('avatar.avatarGender = :avatarGender', {
        avatarGender: dto.avatarGender,
      });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('avatar.avatarName LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }
    const [result, count] = await query
      .orderBy({ 'avatar.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }

  async find(dto: AvatarPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('avatar')
      .select([
        'avatar.id',
        'avatar.avatar',
        'avatar.avatarName',
        'avatar.avatarGender',
        'avatar.createdAt',
      ])
      .where('avatar.status = :status', {
        status: DefaultStatus.ACTIVE,
      });
    if (dto.avatarGender && dto.avatarGender.length > 0) {
      query.andWhere('avatar.avatarGender = :avatarGender', {
        avatarGender: dto.avatarGender,
      });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('avatar.avatarName LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }
    const [result, count] = await query
      .orderBy({ 'avatar.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Avatar Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateAvatarDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Avatar not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async image(image: string, result: Avatar) {
    if (result.avatarPath) {
      const oldPath = join(__dirname, '..', '..', result.avatarPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      avatar: process.env.CLU_CDN_LINK + image,
      avatarPath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Avatar not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }
}
