import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CasePaginationDto, CreateCaseDto } from './dto/create-case.dto';
import { UpdateCaseDto } from './dto/update-case.dto';
import { Case } from './entities/case.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, In, Not, Repository } from 'typeorm';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { UserCase } from 'src/user-case/entities/user-case.entity';
import { CasePassStatus } from 'src/enum';

@Injectable()
export class CaseService {
  constructor(
    @InjectRepository(Case) private readonly repo: Repository<Case>,
    @InjectRepository(UserCase)
    private readonly userCaseRepo: Repository<UserCase>,
  ) {}

  async create(dto: CreateCaseDto) {
    const result = await this.repo.findOne({
      where: { gameId: dto.gameId, question: dto.question },
    });
    if (result) {
      throw new ConflictException('Question already exists in this game!');
    }
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findAll(gameId: string, dto: CasePaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('case')
      .where('case.gameId = :gameId', { gameId: gameId });
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'case.question LIKE :keyword OR case.ans1 LIKE :keyword OR case.ans2 LIKE :keyword OR case.ans3 LIKE :keyword OR case.ans4 LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy('case.createdAt', 'DESC')
      .limit(dto.limit)
      .offset(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findAllUser(gameId: string, accountId: string) {
    const solvedCases = await this.userCaseRepo.find({
      where: {
        accountId,
        gameId,
        casePassStatus: In([
          CasePassStatus.ELEMINATED,
          CasePassStatus.ATTEMPTED,
        ]),
      },
      select: ['caseId'],
    });
    const solvedCaseIds = solvedCases.map((e) => e.caseId);
    const query = this.repo
      .createQueryBuilder('case')
      .select([
        'case.id',
        'case.gameId',
        'case.question',
        'case.qImage',
        'case.hint',
        'case.ans1',
        'case.ans2',
        'case.ans3',
        'case.ans4',
        'case.ans1Image',
        'case.ans2Image',
        'case.ans3Image',
        'case.ans4Image',
        'case.correctAnsNum',
        'case.winPoint',
        'case.createdAt',
      ])
      .where('case.gameId = :gameId', { gameId });
    if (solvedCaseIds.length > 0) {
      query.andWhere('case.id NOT IN (:...solvedCaseIds)', { solvedCaseIds });
    }
    const [result, total] = await query.getManyAndCount();

    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Case Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateCaseDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Not Found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async qImage(image: string, result: Case) {
    if (result.qImagePath) {
      const oldPath = join(__dirname, '..', '..', result.qImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      qImage: process.env.CLU_CDN_LINK + image,
      qImagePath: image,
    });
    return this.repo.save(obj);
  }

  async ans1Image(image: string, result: Case) {
    if (result.ans1ImagePath) {
      const oldPath = join(__dirname, '..', '..', result.ans1ImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      ans1Image: process.env.CLU_CDN_LINK + image,
      ans1ImagePath: image,
    });
    return this.repo.save(obj);
  }

  async ans2Image(image: string, result: Case) {
    if (result.ans2ImagePath) {
      const oldPath = join(__dirname, '..', '..', result.ans2ImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      ans2Image: process.env.CLU_CDN_LINK + image,
      ans2ImagePath: image,
    });
    return this.repo.save(obj);
  }

  async ans3Image(image: string, result: Case) {
    if (result.ans3ImagePath) {
      const oldPath = join(__dirname, '..', '..', result.ans3ImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      ans3Image: process.env.CLU_CDN_LINK + image,
      ans3ImagePath: image,
    });
    return this.repo.save(obj);
  }

  async ans4Image(image: string, result: Case) {
    if (result.ans4ImagePath) {
      const oldPath = join(__dirname, '..', '..', result.ans4ImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      ans4Image: process.env.CLU_CDN_LINK + image,
      ans4ImagePath: image,
    });
    return this.repo.save(obj);
  }

  async ansImage(files: Express.Multer.File[], result: Case) {
    const oldImages = [
      result.ans1ImagePath,
      result.ans2ImagePath,
      result.ans3ImagePath,
      result.ans4ImagePath,
    ];
    for (const oldPath of oldImages) {
      if (oldPath) {
        try {
          const fullPath = join(__dirname, '..', '..', oldPath);
          await unlink(fullPath);
        } catch (err) {
          console.warn(`Failed to delete old image: ${oldPath}`, err.message);
        }
      }
    }
    const imageFields = ['ans1Image', 'ans2Image', 'ans3Image', 'ans4Image'];
    const imagePathFields = [
      'ans1ImagePath',
      'ans2ImagePath',
      'ans3ImagePath',
      'ans4ImagePath',
    ];

    files.forEach((file, index) => {
      if (file && imageFields[index]) {
        result[imageFields[index]] = process.env.CLU_CDN_LINK + file.path;
        result[imagePathFields[index]] = file.path;
      }
    });

    return this.repo.save(result);
  }

  async remove(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Not Found!');
    }
    return this.repo.remove(result);
  }
}
