import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreateCategoryDto } from './dto/create-category.dto';
import { UpdateCategoryDto } from './dto/update-category.dto';
import { Category } from './entities/category.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { DefaultStatusPaginationDto } from 'src/common/dto/default-status-pagination.dto';
import { DefaultStatus } from 'src/enum';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';

@Injectable()
export class CategoryService {
  constructor(
    @InjectRepository(Category) private readonly repo: Repository<Category>,
  ) {}

  async create(dto: CreateCategoryDto) {
    const result = await this.repo.findOne({
      where: { name: dto.name },
    });
    if (result) {
      throw new ConflictException('Already exists!');
    }
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: DefaultStatusPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('category');
    if (dto.status && dto.status.length > 0) {
      query.where('category.status = :status', { status: dto.status });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'category.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findByUser() {
    const query = await this.repo
      .createQueryBuilder('category')
      .where('category.status = :status', {
        status: DefaultStatus.ACTIVE,
      });
    const [result, total] = await query
      // .orderBy('category.createdAt', 'ASC')
      .getManyAndCount();

    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateCategoryDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    const obj = Object.assign(result, dto);
    return await this.repo.save(obj);
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    const obj = Object.assign(result, dto);
    return await this.repo.save(obj);
  }
}
