import {
  BadRequestException,
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { Brackets, Repository } from 'typeorm';
import { CityDto, PaginationSDto, UpdateCityDto } from './dto/city.dto';
import { City } from './entities/city.entity';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import { DefaultStatus } from 'src/enum';
import { unlink } from 'fs/promises';
import { join } from 'path';

@Injectable()
export class CityService {
  constructor(
    @InjectRepository(City)
    private readonly repo: Repository<City>,
  ) {}

  async create(image: string, dto: CityDto) {
    const city = await this.repo.findOne({
      where: { name: dto.name },
    });
    if (city) {
      throw new ConflictException('This city already exists!');
    }
    const obj = Object.assign({
      image: process.env.CLU_CDN_LINK + image,
      imagePath: image,
      name: dto.name,
    });
    return this.repo.save(obj);
  }

  async findAll(dto: PaginationSDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('city');
    query.where('city.status = :status', {
      status: dto.status,
    });
    const [result, count] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where('city.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN city.name LIKE '${keyword}%' THEN 0 ELSE 1 END, city.name`,
        'ASC',
      )
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }

  async find(dto: CommonPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('city');
    query.where('city.status = :status', { status: DefaultStatus.ACTIVE });
    const [result, count] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where('city.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN city.name LIKE '${keyword}%' THEN 0 ELSE 1 END, city.name`,
        'ASC',
      )
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('City Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateCityDto) {
    try {
      const city = await this.repo.findOne({ where: { id } });
      if (!city) {
        throw new NotFoundException('City not found!');
      }
      const obj = Object.assign(city, { name: dto.name });
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either city exists or invalid name!',
      );
    }
  }

  async image(image: string, result: City) {
    if (result.imagePath) {
      const oldPath = join(__dirname, '..', '..', result.imagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      image: process.env.CLU_CDN_LINK + image,
      imagePath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: DefaultStatusDto) {
    const city = await this.repo.findOne({ where: { id } });
    if (!city) {
      throw new NotFoundException('City not found!');
    }
    const obj = Object.assign(city, dto);
    return this.repo.save(obj);
  }
}
