import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { CreateFeedbackModuleDto } from './dto/create-feedback-module.dto';
import { UpdateFeedbackModuleDto } from './dto/update-feedback-module.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { FeedbackModule } from './entities/feedback-module.entity';
import { Like, Repository } from 'typeorm';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';

@Injectable()
export class FeedbackModuleService {
  constructor(
    @InjectRepository(FeedbackModule)
    private readonly repo: Repository<FeedbackModule>,
  ) {}

  async create(dto: CreateFeedbackModuleDto) {
    try {
      const result = await this.repo.findOne({
        where: { moduleName: dto.moduleName },
      });
      if (result) {
        throw new ConflictException('Module Name already exists!');
      }
      const obj = Object.assign(dto);
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async findAll(dto: CommonPaginationDto) {
    try {
      const keyword = dto.keyword || '';
      const [result, total] = await this.repo.findAndCount({
        where: {
          moduleName: Like('%' + keyword + '%'),
        },
        order: { createdAt: 'DESC' },
        take: dto.limit,
        skip: dto.offset,
      });
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async find() {
    try {
      const [result, total] = await this.repo.findAndCount({
        select: ['id', 'moduleName', 'createdAt'],
        order: { createdAt: 'DESC' },
      });
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async update(id: string, dto: CreateFeedbackModuleDto) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Module Name not found!');
      }
      const obj = Object.assign(result, dto);
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async remove(id: string) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Module Name not found!');
      }
      return this.repo.remove(result);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }
}
