import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  Put,
} from '@nestjs/common';
import { FeedbackService } from './feedback.service';
import {
  CreateFeedbackDto,
  FeedbackPaginationDto,
  FeedbackStatusDto,
} from './dto/create-feedback.dto';
import { UpdateFeedbackDto } from './dto/update-feedback.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';

@Controller('feedback')
export class FeedbackController {
  constructor(private readonly feedbackService: FeedbackService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  create(@Body() dto: CreateFeedbackDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.feedbackService.create(dto);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN)
  // @CheckPermissions([PermissionAction.READ, 'feedback'])
  findAll(@Query() dto: FeedbackPaginationDto) {
    return this.feedbackService.findAll(dto);
  }

  @Get('user/list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  find(@Query() dto: CommonPaginationDto, @CurrentUser() user: Account) {
    return this.feedbackService.find(dto, user.id);
  }

  @Put('resolve/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  statusByUser(@Param('id') id: string) {
    return this.feedbackService.statusByUser(id);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  status(@Param('id') id: string, @Body() dto: FeedbackStatusDto) {
    return this.feedbackService.status(id, dto);
  }

  @Delete('remove/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  remove(@Param('id') id: string) {
    return this.feedbackService.remove(id);
  }
}
