import {
  BadRequestException,
  Body,
  Controller,
  Get,
  HttpException,
  HttpStatus,
  NotAcceptableException,
  Param,
  Post,
  Query,
  Res,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { Response } from 'express';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PaymentStatus, PaymentType, UserRole } from 'src/enum';
// import { NotifyService } from 'src/notify/notify.service';
import { checkPaymentStatus, generateUrl } from 'src/utils/payment.util';
import { instance } from 'src/utils/razor-pay.utils';
import {
  OrderDto,
  PaginationDto,
  PayDto,
  PaymentHistoryDto,
  verifyPaymentDto,
} from './dto/payment-history.dto';
import { PaymentHistoryService } from './payment-history.service';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';

@Controller('payment-history')
export class PaymentHistoryController {
  constructor(
    private readonly paymentHistoryService: PaymentHistoryService,
    // private readonly notifyService: NotifyService,
    private readonly nodeMailerService: NodeMailerService,
  ) {}

  @Post('order')
  // @UseGuards(AuthGuard('jwt'))
  async createOrder(@Body() dto: OrderDto) {
    // return await this.paymentHistoryService.createOrder(dto);
  }

  @Post('save-payment')
  @UseGuards(AuthGuard('jwt'))
  async savePayment(
    @Body() dto: verifyPaymentDto,
    @CurrentUser() user: Account,
  ) {
    return this.paymentHistoryService.savePayment(dto, user.id);
  }

  @Get('paymentFailed-mail/:businessAccId')
  @UseGuards(AuthGuard('jwt'))
  async paymentFailed(
    @Param('businessAccId') businessAccId: string,
    @CurrentUser() user: Account,
  ) {
    if (!businessAccId) {
      throw new BadRequestException('Business AccountId required!');
    }
    const response = await this.paymentHistoryService.paymentFailed(
      user.id,
      businessAccId,
    );
    if (!response.success) {
      throw new HttpException(
        response.message,
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    }
    return response;
  }

  @Get('all/list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  // @Roles(UserRole.BUSINESS)
  findAll(@Query() dto: PaginationDto, @CurrentUser() user: Account) {
    return this.paymentHistoryService.find(dto, user.id);
  }

  @Post('pay')
  async payDoctor(@Body() dto, @Res() res: Response) {
    const result = await checkPaymentStatus(dto.transactionId);
    if (!result.success) {
      return res.redirect(
        process.env.STN_REDIRECT_URL + '/failed/' + dto.transactionId,
      );
    } else {
      return res.redirect(
        process.env.STN_REDIRECT_URL + '/success/' + dto.transactionId,
      );
    }
  }

  @Get('admin/list/total')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  findAllByAdminTotal(@Query() dto: PaginationDto) {
    return this.paymentHistoryService.findTotal(dto);
  }

  @Get('member/payment-list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  paymentList(@Query() dto: CommonPaginationDto, @CurrentUser() user: Account) {
    return this.paymentHistoryService.paymentList(dto, user.id);
  }

  @Get('payment-detail/member/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async getPaymentDetail(@Param('id') id: string, @Res() res: Response) {
    const paymentDetail = await this.paymentHistoryService.getPaymentDetail(id);
    const invoice = {
      transactionId: paymentDetail.transactionId,
      invoiceNumber: paymentDetail.invoiceNumber,
      paymentDate: new Date(paymentDetail.createdAt).toLocaleDateString(),
      mode: 'Online',
      amount: paymentDetail.amount,
      total: paymentDetail.total,
      membershipCardName: paymentDetail.userDetail['membershipCard']['name'],
      duration: paymentDetail.userDetail['membershipCard']['validity'],
      userDetail: {
        billTo: paymentDetail.userDetail['billTo'],
        name:
          paymentDetail.userDetail['fName'] +
          ' ' +
          `${paymentDetail.userDetail['mName'] || null}` +
          ' ' +
          paymentDetail.userDetail['lName'],
        memberId: paymentDetail.userDetail['memberId'],
        address: paymentDetail.userDetail['address1'],
        from: paymentDetail.userDetail['membershipValidFrom'],
        to: paymentDetail.userDetail['membershipValidTo'],
        businessName: paymentDetail.userDetail['businessName'],
        gstNumber: paymentDetail.userDetail['gstNumber'],
        businessAddress1: paymentDetail.userDetail['businessAddress1'],
      },
      business: {
        businessName: paymentDetail.account['business']['0']['businessName'],
        gstNo: paymentDetail.account['business']['0']['gstNo'],
        address: paymentDetail.account['business']['0']['address1'],
      },
      tax: paymentDetail.account['tax'],
    };

    // const pdf = await createInvoice(invoice);
    // const name =
    //   Date.now().toString() +
    //   `-invoice_${paymentDetail.userDetail['memberId']}.pdf`;
    // res.setHeader('Content-Type', 'application/pdf');
    // res.set('Content-Disposition', `attachment; filename="${name}"`);
    // pdf.pipe(res);
    // pdf.end();
  }

  @Get('invoice-mail/:id/:businessAccId')
  @UseGuards(AuthGuard('jwt'))
  @Roles(UserRole.USER)
  async userPaymentInvoiceMail(
    @Param('id') id: string,
    @Param('businessAccId') businessAccId: string,
    @Res() res: Response,
  ) {
    if (!businessAccId) {
      throw new BadRequestException('Business AccountId required!');
    }
    const paymentDetail = await this.paymentHistoryService.getPaymentDetail(id);
    const invoice = {
      transactionId: paymentDetail.transactionId,
      invoiceNumber: paymentDetail.invoiceNumber,
      paymentDate: new Date(paymentDetail.createdAt).toLocaleDateString(),
      mode: 'Online',
      amount: paymentDetail.amount,
      total: paymentDetail.total,
      membershipCardName: paymentDetail.userDetail['membershipCard']['name'],
      duration: paymentDetail.userDetail['membershipCard']['validity'],
      userDetail: {
        billTo: paymentDetail.userDetail['billTo'],
        name:
          paymentDetail.userDetail['fName'] +
          ' ' +
          `${paymentDetail.userDetail['mName'] || null}` +
          ' ' +
          paymentDetail.userDetail['lName'],
        memberId: paymentDetail.userDetail['memberId'],
        address: paymentDetail.userDetail['address1'],
        from: paymentDetail.userDetail['membershipValidFrom'],
        to: paymentDetail.userDetail['membershipValidTo'],
        businessName: paymentDetail.userDetail['businessName'],
        gstNumber: paymentDetail.userDetail['gstNumber'],
        businessAddress1: paymentDetail.userDetail['businessAddress1'],
      },
      business: {
        businessName: paymentDetail.account['business']['0']['businessName'],
        gstNo: paymentDetail.account['business']['0']['gstNo'],
        address: paymentDetail.account['business']['0']['address1'],
      },
      tax: paymentDetail.account['tax'],
    };

    // const pdfDoc = await createInvoice(invoice);

    // const buffers: Buffer[] = [];
    // pdfDoc.on('data', buffers.push.bind(buffers));
    // pdfDoc.on('end', async () => {
    //   const pdfBuffer = Buffer.concat(buffers);
    //   const emailSent = await this.nodeMailerService.userPaymentInvoice(
    //     paymentDetail.userDetail['email'],
    //     paymentDetail.userDetail['memberId'],
    //     pdfBuffer,
    //     businessAccId,
    //   );
    //   if (emailSent) {
    //     return res.status(200).json({
    //       success: true,
    //       message: 'Invoice sent to email successfully.',
    //     });
    //   } else {
    //     return res
    //       .status(500)
    //       .json({ success: false, message: 'Failed to send invoice email.' });
    //   }
    // });

    // pdfDoc.end();
  }
}
