import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import {
  DefaultStatus,
  PaymentStatus,
  PaymentType,
  SubscriptionStatus,
} from 'src/enum';
// import { NodeMailerService } from "src/node-mailer/node-mailer.service";
// import { NotifyService } from 'src/notify/notify.service';
import { Brackets, Repository } from 'typeorm';
import {
  OrderDto,
  PaginationDto,
  PayDto,
  PaymentHistoryDto,
  PhonePayHistoryDto,
  verifyPaymentDto,
} from './dto/payment-history.dto';
import { PaymentHistory } from './entities/payment-history.entity';
// import { NotifyService } from 'src/notify/notify.service';
import Razorpay from 'razorpay';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
// const crypto = require('crypto');

@Injectable()
export class PaymentHistoryService {
  constructor(
    @InjectRepository(PaymentHistory)
    private readonly paymentRepo: Repository<PaymentHistory>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
    // private readonly notifyService: NotifyService,
    private readonly nodeMailerService: NodeMailerService,
  ) {}

  private async getRazorpayInstance(accountId: string) {
    // const business = await this.bpRepo.findOne({
    //   where: { accountId: accountId },
    // });
    // if (!business || !business.keyId || !business.keySecret) {
    //   throw new NotFoundException('Business or Razorpay credentials not found');
    // }
    // return new Razorpay({
    //   key_id: business.keyId,
    //   key_secret: business.keySecret,
    // });
  }

  // async createOrder(dto: OrderDto) {
  //   const razorpay = await this.getRazorpayInstance(dto.accountId);
  //   const options = {
  //     amount: +dto.amount * 100,
  //     currency: 'INR',
  //     receipt: dto.email,
  //     payment_capture: '0',
  //   };
  //   const payload = await razorpay.orders.create(options);
  //   if (payload.status === 'created') {
  //     return payload;
  //   } else {
  //     throw new NotAcceptableException(
  //       'Try after some time or user other payment methods!',
  //     );
  //   }
  // }

  async savePayment(dto: verifyPaymentDto, userAccountId: string) {
    const check = await this.paymentRepo.findOne({
      where: {
        paymentId: dto.paymentId,
        signature: dto.signature,
        orderId: dto.orderId,
      },
    });
    if (check) {
      throw new ConflictException('Please make a new payment!');
    }
    const userDetail = await this.udRepo.findOne({
      where: { accountId: userAccountId },
    });
    const transactionId = Math.floor(1000000000 + Math.random() * 9000000000);
    dto.transactionId = transactionId.toString();
    dto.userDetailId = userDetail.id;
    dto.invoiceNumber = '#' + Date.now().toString();

    if (dto.status == 'Success') {
      // const membershipCard = await this.memCardRepo.findOne({
      //   where: { id: userDetail.membershipCardId },
      // });
      // if (!membershipCard) {
      //   throw new NotFoundException('Please select a membership card first!');
      // }

      dto.status = PaymentStatus.COMPLETED;
      const obj = Object.assign(dto);
      const payment = await this.paymentRepo.save(obj);

      const lastUser = await this.udRepo.findOne({
        // where: { createdById: userDetail.createdById },
      });
      let nextNumber = 1;
      const paddedNumber = String(nextNumber).padStart(3, '0');
      const memberId = `M-${paddedNumber}-M`;
      const cardNumber = `CRD-${Math.floor(1000 + Math.random() * 9000)}`;

      const today = new Date();
      // const duration = parseInt(membershipCard.validity);
      const startDate = new Date();
      const endDate = new Date(today);
      // endDate.setDate(today.getDate() + duration - 1);

      await this.udRepo.save(userDetail);

      return payment;
    } else {
      throw new ConflictException('Payment failed!');
    }
  }

  async paymentFailed(userAccountId: string, businessAccId: string) {
    try {
      const userDetail = await this.udRepo.findOne({
        where: { accountId: userAccountId },
      });
      if (!userDetail) {
        throw new NotFoundException('User not found!');
      }
      return { success: true, message: 'mail send successfully' };
    } catch (error) {
      console.error('Error in userPaymentFailed:', error);
      if (error instanceof NotFoundException) throw error;
      return {
        success: false,
        message: 'Internal server error while sending mail.',
      };
    }
  }

  async findAll(dto: PaginationDto, accountId: string) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const [result, total] = await this.paymentRepo
      .createQueryBuilder('paymentHistory')
      .leftJoinAndSelect('paymentHistory.account', 'account')
      .where(
        'paymentHistory.createdAt >= :fromDate AND paymentHistory.createdAt <= :toDate AND paymentHistory.accountId = :accountId',
        {
          fromDate: fromDate,
          toDate: toDate,
          accountId: accountId,
        },
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'paymentHistory.createdAt': 'DESC' })
      .getManyAndCount();
    return { result, total };
  }

  async find(dto: PaginationDto, businessAccId: string) {
    const keyword = dto.keyword || '';
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);

    const query = await this.paymentRepo
      .createQueryBuilder('paymentHistory')
      // .leftJoinAndSelect('paymentHistory.account', 'account')
      .leftJoinAndSelect('paymentHistory.userDetail', 'userDetail')
      .select([
        'paymentHistory.id',
        'paymentHistory.userDetailId',
        'paymentHistory.transactionId',
        'paymentHistory.invoiceNumber',
        'paymentHistory.orderId',
        'paymentHistory.signature',
        'paymentHistory.paymentId',
        'paymentHistory.amount',
        'paymentHistory.total',
        'paymentHistory.status',
        'paymentHistory.mode',
        'paymentHistory.createdAt',
        'paymentHistory.accountId',

        // 'account.id',
        // 'account.phoneNumber',
        // 'account.email',

        'userDetail.id',
        'userDetail.email',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.memberId',
        'userDetail.cardNumber',
        'userDetail.membershipCardId',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',
      ])
      .where(
        'paymentHistory.accountId = :businessAccId AND paymentHistory.createdAt >= :fromDate AND paymentHistory.createdAt <= :toDate',
        {
          businessAccId: businessAccId,
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    if (dto.userDetailId && dto.userDetailId.length > 0) {
      query.andWhere('paymentHistory.userDetailId = :userDetailId', {
        userDetailId: dto.userDetailId,
      });
    }

    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'userDetail.fName LIKE :fName OR paymentHistory.invoiceNumber LIKE :invoiceNumber OR paymentHistory.orderId LIKE :orderId',
            {
              fName: '%' + keyword + '%',
              invoiceNumber: '%' + keyword + '%',
              orderId: '%' + keyword + '%',
            },
          );
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'paymentHistory.createdAt': 'DESC' })
      .getManyAndCount();
    return { result, total };
  }

  async paymentList(dto: CommonPaginationDto, userAccountId: string) {
    const user = await this.udRepo.findOne({
      where: { accountId: userAccountId },
    });
    if (!user) {
      throw new NotFoundException('User Not Found!');
    }
    const [result, total] = await this.paymentRepo.findAndCount({
      select: [
        'id',
        'transactionId',
        'amount',
        'total',
        'status',
        'mode',
        'createdAt',
        'userDetailId',
        'accountId',
      ],
      where: { userDetailId: user.id },
      take: dto.limit,
      skip: dto.offset,
    });
    return { result, total };
  }

  async getPaymentDetail(id: string) {
    const result = await this.paymentRepo
      .createQueryBuilder('paymentHistory')
      .leftJoinAndSelect('paymentHistory.account', 'account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect('account.tax', 'tax', 'tax.status = :status', {
        status: DefaultStatus.ACTIVE,
      })
      .leftJoinAndSelect('paymentHistory.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'paymentHistory.id',
        'paymentHistory.transactionId',
        'paymentHistory.invoiceNumber',
        'paymentHistory.amount',
        'paymentHistory.total',
        'paymentHistory.status',
        'paymentHistory.mode',
        'paymentHistory.createdAt',

        'account.id',
        'business.id',
        'business.businessName',
        'business.parentCompanyName',
        'business.gstNo',
        'business.address1',
        'business.address2',

        'tax.id',
        'tax.taxName',
        'tax.rate',
        'tax.status',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.email',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.billTo',
        'userDetail.businessName',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessAddress1',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
      ])
      .where('paymentHistory.id = :id', { id: id })
      .getOne();
    return result;
  }

  async findTotal(dto: PaginationDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);
    const total = await this.paymentRepo
      .createQueryBuilder('paymentHistory')
      .select([
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :PENDING THEN paymentHistory.total ELSE 0 END), 0) AS pendingTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :COMPLETED THEN paymentHistory.total ELSE 0 END), 0) AS completedTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :FAILED THEN paymentHistory.total ELSE 0 END), 0) AS failedTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :CANCELLED THEN paymentHistory.total ELSE 0 END), 0) AS cancelledTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :PENDING THEN paymentHistory.wallet ELSE 0 END), 0) AS pendingWalletTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :COMPLETED THEN paymentHistory.wallet ELSE 0 END), 0) AS completedWalletTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :FAILED THEN paymentHistory.wallet ELSE 0 END), 0) AS failedWalletTotal',
        'COALESCE(SUM(CASE WHEN paymentHistory.status = :CANCELLED THEN paymentHistory.wallet ELSE 0 END), 0) AS cancelledWalletTotal',
      ])
      .where(
        'paymentHistory.createdAt >= :fromDate AND paymentHistory.createdAt <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      )
      .setParameters({
        PENDING: 'PENDING',
        COMPLETED: 'COMPLETED',
        FAILED: 'FAILED',
        CANCELLED: 'CANCELLED',
      })
      .groupBy('paymentHistory.status')
      .getRawOne();
    if (total) {
      return total;
    } else {
      return {
        pendingTotal: 0,
        completedTotal: 0,
        failedTotal: 0,
        cancelledTotal: 0,
        pendingWalletTotal: 0,
        completedWalletTotal: 0,
        failedWalletTotal: 0,
        cancelledWalletTotal: 0,
      };
    }
  }
}
