import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
} from '@nestjs/common';
import { UserCaseService } from './user-case.service';
import { CreateUserCaseDto } from './dto/create-user-case.dto';
import { UpdateUserCaseDto } from './dto/update-user-case.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';

@Controller('user-case')
export class UserCaseController {
  constructor(private readonly userCaseService: UserCaseService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  create(@Body() dto: CreateUserCaseDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.userCaseService.create(dto, user.id);
  }

  @Get('preview/:gameId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  resultPreview(
    @Param('gameId') gameId: string,
    @CurrentUser() user: Account,
    @Query() dto: CommonPaginationDto,
  ) {
    return this.userCaseService.resultPreview(gameId, user.id, dto);
  }

  @Get('incomplete-count/:gameId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  getUnplayedCount(
    @Param('gameId') gameId: string,
    @CurrentUser() user: Account,
  ) {
    return this.userCaseService.getUnplayedCount(gameId, user.id);
  }

  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.userCaseService.findOne(+id);
  }

  @Delete(':id')
  remove(@Param('id') id: string) {
    return this.userCaseService.remove(+id);
  }
}
