import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Account } from 'src/account/entities/account.entity';
import { Business } from 'src/business/entities/business.entity';
import { CardAmenity } from 'src/card-amenities/entities/card-amenity.entity';
import { DateFilterDto } from 'src/common/dto/date.dto';
import {
  BusinessStatus,
  DefaultStatus,
  MemberStatus,
  YNStatus,
} from 'src/enum';
import { Licence } from 'src/licence/entities/licence.entity';
import { MembershipCard } from 'src/membership-card/entities/membership-card.entity';
import { UserChild } from 'src/user-child/entities/user-child.entity';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { Repository } from 'typeorm';

@Injectable()
export class DashboardService {
  constructor(
    @InjectRepository(Account)
    private readonly acRepo: Repository<Account>,
    @InjectRepository(Business)
    private readonly businessRepo: Repository<Business>,
    @InjectRepository(Licence)
    private readonly licenceRepo: Repository<Licence>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
    @InjectRepository(UserChild)
    private readonly childRepo: Repository<UserChild>,
    @InjectRepository(MembershipCard)
    private readonly mCardRepo: Repository<MembershipCard>,
    @InjectRepository(CardAmenity)
    private readonly amenityRepo: Repository<CardAmenity>,
  ) {}

  async businessCount(dto: DateFilterDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);

    const query1 = await this.businessRepo
      .createQueryBuilder('business')
      .innerJoinAndSelect(
        'business.licence',
        'licence',
        'licence.status = :status',
        { status: DefaultStatus.ACTIVE },
      )
      .where('business.status = :status', { status: BusinessStatus.ACTIVE });
    if (dto.fromDate && dto.toDate) {
      query1.andWhere(
        'licence.startDate >= :fromDate AND licence.renewalDate <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }
    const activeBusiness = await query1.getCount();

    const query2 = await this.businessRepo
      .createQueryBuilder('business')
      .innerJoinAndSelect(
        'business.licence',
        'licence',
        'licence.status = :status',
        { status: DefaultStatus.ACTIVE },
      )
      .where('business.status = :status', { status: BusinessStatus.RENEWAL });
    if (dto.fromDate && dto.toDate) {
      query2.andWhere(
        'licence.startDate >= :fromDate AND licence.renewalDate <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }
    const renewalBusiness = await query2.getCount();

    const query3 = await this.businessRepo
      .createQueryBuilder('business')
      .innerJoinAndSelect(
        'business.licence',
        'licence',
        'licence.status = :status',
        { status: DefaultStatus.ACTIVE },
      )
      .where('business.status = :status', { status: BusinessStatus.EXPERIED });
    if (dto.fromDate && dto.toDate) {
      query3.andWhere(
        'licence.startDate >= :fromDate AND licence.renewalDate <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }
    const expiredBusiness = await query3.getCount();

    const query4 = await this.licenceRepo
      .createQueryBuilder('licence')
      .where('licence.amc = :amc', { amc: YNStatus.YES });
    if (dto.fromDate && dto.toDate) {
      query4.andWhere(
        'licence.startDate >= :fromDate AND licence.renewalDate <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }
    const amc = await query4.getCount();

    return { activeBusiness, renewalBusiness, expiredBusiness, amc };
  }

  async getCount(accountId: string) {
    const memberCount = await this.udRepo.count({
      where: { createdById: accountId },
    });

    const activeMember = await this.udRepo.count({
      where: { createdById: accountId, status: MemberStatus.ACTIVE },
    });

    const cardCount = await this.mCardRepo.count({ where: { accountId } });

    const expiredMember = await this.udRepo.count({
      where: { createdById: accountId, status: MemberStatus.EXPIRED },
    });

    const totalStaff = await this.acRepo.count({
      where: { businessAdminId: accountId },
    });

    const totalchild = await this.childRepo.count({
      where: { businessAccId: accountId },
    });

    // const totalAmenities = await this.amenityRepo.count({
    //   where: { accountId: accountId },
    // });
    const totalAmenities = await this.amenityRepo
      .createQueryBuilder('amenity')
      .select('COUNT(DISTINCT amenity.name)', 'count')
      .where('amenity.accountId = :accountId', { accountId })
      .getRawOne();
    const count = parseInt(totalAmenities.count, 10);

    const pendingMember = await this.udRepo.count({
      where: { createdById: accountId, status: MemberStatus.PENDING },
    });

    const enquiries = await this.udRepo.count({
      where: { createdById: accountId, status: MemberStatus.ENQUIRY },
    });

    return {
      memberCount: memberCount,
      activeMembership: activeMember,
      cardCount: cardCount,
      expiredMember: expiredMember,
      totalStaff: totalStaff,
      totalchild: totalchild,
      totalAmenities: count,
      pendingMember: pendingMember,
      totalEnquiries: enquiries,
    };
  }
}
