import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreateFloaterDto } from './dto/create-floater.dto';
import { Floater } from './entities/floater.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Like, Repository } from 'typeorm';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';

@Injectable()
export class FloaterService {
  constructor(
    @InjectRepository(Floater) private readonly repo: Repository<Floater>,
  ) {}

  async create(dto: CreateFloaterDto) {
    const result = await this.repo.findOne({
      where: { message: dto.message, accountId: dto.accountId },
    });
    if (result) {
      throw new ConflictException(
        'This floating message already exists for this business!',
      );
    }
    const find = await this.repo.find({
      where: { accountId: dto.accountId },
    });
    if (find.length) {
      await this.repo.remove(find);
    }

    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async find(accountId: string) {
    const result = await this.repo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException(
        'No floating message found for this business',
      );
    }
    return result;
  }

  async findByBusiness(accountId: string) {
    return this.repo.findOne({ where: { accountId } });
  }

  async findAll(dto: CommonPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('floater')
      .leftJoinAndSelect('floater.account', 'account')
      .leftJoinAndSelect('account.business', 'business')
      .select([
        'floater.id',
        'floater.message',
        'floater.createdAt',

        'account.id',
        'account.email',

        'business.businessName',
      ]);
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('floater.message LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }

    const [result, total] = await query
      .orderBy({ 'floater.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async remove(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('No floating message found');
    }
    return this.repo.remove(result);
  }
}
