import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import {
  MemberStatusDto,
  UpdateMemberDto,
  UpdateUserDetailDto,
  UpdateUserProfileDto,
} from './dto/update-user-details';
import { UserDetail } from './entities/user-detail.entity';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import { MembershipCard } from 'src/membership-card/entities/membership-card.entity';
import { DefaultStatus, MemberStatus } from 'src/enum';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { unlink } from 'fs/promises';
import { join } from 'path';

@Injectable()
export class UserDetailsService {
  constructor(
    @InjectRepository(UserDetail) private readonly repo: Repository<UserDetail>,
    @InjectRepository(MembershipCard)
    private readonly memCardRepo: Repository<MembershipCard>,
    private readonly nodeMailerService: NodeMailerService,
  ) {}

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { accountId: id } });
    if (!result) {
      throw new NotFoundException('User not found!');
    }
    return result;
  }

  async sendCardDetailMail(accountId: string, businessAccId: string) {
    try {
      const result = await this.repo.findOne({ where: { accountId } });
      if (!result) {
        throw new NotFoundException('User not found!');
      }
      const card = await this.memCardRepo.findOne({
        where: { id: result.membershipCardId },
      });
      if (!card) {
        throw new NotFoundException('Card not selected!');
      }
      const emailSent = await this.nodeMailerService.sendMembershipCardDetail(
        result.email,
        result.cardNumber,
        card.name,
        result.membershipValidFrom,
        result.membershipValidTo,
        businessAccId,
      );
      if (!emailSent) {
        return { success: false, message: 'Failed to send mail!' };
      }
      return { success: true, message: 'Mail sent successfully' };
    } catch (error) {
      console.error('Error in sendCardDetailMail:', error);
      if (error instanceof NotFoundException) throw error;
      return {
        success: false,
        message: 'Internal server error while sending mail.',
      };
    }
  }

  async update(dto: UpdateUserDetailDto, accountId: string) {
    const result = await this.repo.findOne({ where: { accountId: accountId } });
    if (!result) {
      throw new NotFoundException('User profile not found!');
    }
    const membershipCard = await this.memCardRepo.findOne({
      where: { id: dto.membershipCardId },
    });
    if (!membershipCard) {
      throw new NotFoundException('MembershipCard not found!');
    }
    // const today = new Date();
    // const startDate = new Date().toLocaleDateString('en-CA');
    // const duration = parseInt(membershipCard.validity);
    // const endDate = new Date(today);
    // endDate.setDate(today.getDate() + duration - 1);
    // const endDateString = endDate.toLocaleDateString('en-CA');

    const obj = Object.assign(result, {
      membershipCardId: dto.membershipCardId,
      salutation: dto.salutation,
      email: dto.email,
      fName: dto.fName,
      mName: dto.mName,
      lName: dto.lName,
      gender: dto.gender,
      address1: dto.address1,
      address2: dto.address2,
      city: dto.city,
      state: dto.state,
      zipcode: dto.zipcode,
      aadharNumber: dto.aadharNumber,
      haveBusiness: dto.haveBusiness,
      businessType: dto.businessType,
      businessName: dto.businessName,
      gstNumber: dto.gstNumber,
      businessCity: dto.businessCity,
      businessState: dto.businessState,
      businessZipcode: dto.businessZipcode,
      businessPhone: dto.businessPhone,
      businessEmail: dto.businessEmail,
      landMark: dto.landMark,
      businessLandmark: dto.businessLandmark,
      fatherName: dto.fatherName,
      dob: dto.dob,
      qualification: dto.qualification,
      profession: dto.profession,
      panNumber: dto.panNumber,
      income: dto.income,
      businessAddress1: dto.businessAddress1,
      businessAddress2: dto.businessAddress2,
      maritalStatus: dto.maritalStatus,
      billTo: dto.billTo,
      businessTurnover: dto.businessTurnover,

      cardNumber: null,
      membershipValidFrom: null,
      membershipValidTo: null,
      memberId: null,
    });
    return this.repo.save(obj);
  }

  async profileUpdate(dto: UpdateUserProfileDto, accountId: string) {
    const result = await this.repo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('User Not Found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async renewal(accountId: string, membershipCardId: string) {
    const user = await this.repo.findOne({ where: { accountId: accountId } });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    if (user.status != MemberStatus.ACTIVE) {
      throw new ConflictException('Account is not active!');
    }
    const membershipCard = await this.memCardRepo.findOne({
      where: { id: membershipCardId },
    });
    if (!membershipCard) {
      throw new NotFoundException('Card not found!');
    }
    // const today = new Date();
    // const startDate = new Date().toLocaleDateString('en-CA');
    // const duration = parseInt(membershipCard.validity);
    // const endDate = new Date(today);
    // endDate.setDate(today.getDate() + duration - 1);
    // const endDateString = endDate.toLocaleDateString('en-CA');

    const obj = Object.assign(user, {
      membershipCardId: membershipCardId,
      membershipValidFrom: null,
      membershipValidTo: null,
    });
    return this.repo.save(obj);
  }

  async updateMember(accountId: string, dto: UpdateMemberDto) {
    const result = await this.repo.findOne({ where: { accountId: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async profileImage(image: string, result: UserDetail) {
    if (result.profilePath) {
      const oldPath = join(__dirname, '..', '..', result.profilePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      profile: process.env.PV_CDN_LINK + image,
      profilePath: image,
    });
    return this.repo.save(obj);
  }

  async memberDoc(image: string, result: UserDetail) {
    if (result.memberDocPath) {
      const oldPath = join(__dirname, '..', '..', result.memberDocPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      memberDoc: process.env.PV_CDN_LINK + image,
      memberDocPath: image,
    });
    return this.repo.save(obj);
  }

  async businessDoc(image: string, result: UserDetail) {
    if (result.businessDocPath) {
      const oldPath = join(__dirname, '..', '..', result.businessDocPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      businessDoc: process.env.PV_CDN_LINK + image,
      businessDocPath: image,
    });
    return this.repo.save(obj);
  }

  async pan(image: string, result: UserDetail) {
    if (result.panPath) {
      const oldPath = join(__dirname, '..', '..', result.panPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      pan: process.env.PV_CDN_LINK + image,
      panPath: image,
    });
    return this.repo.save(obj);
  }

  async aadhar(image: string, result: UserDetail) {
    if (result.aadharPath) {
      const oldPath = join(__dirname, '..', '..', result.aadharPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      aadhar: process.env.PV_CDN_LINK + image,
      aadharPath: image,
    });
    return this.repo.save(obj);
  }

  async adminFile(image: string, result: UserDetail) {
    if (result.adminFilePath) {
      const oldPath = join(__dirname, '..', '..', result.adminFilePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      adminFile: process.env.PV_CDN_LINK + image,
      adminFilePath: image,
    });
    return this.repo.save(obj);
  }

  async regFile(image: string, result: UserDetail) {
    if (result.regFilePath) {
      const oldPath = join(__dirname, '..', '..', result.regFilePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      regFile: process.env.PV_CDN_LINK + image,
      regFilePath: image,
    });
    return this.repo.save(obj);
  }

  async memberStatus(accountId: string, dto: MemberStatusDto) {
    const result = await this.repo.findOne({ where: { accountId: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, { status: dto.status });
    const status = await this.repo.save(obj);
    if (dto.status == MemberStatus.APPROVED) {
      this.nodeMailerService.userAccountApproval(
        result.email,
        result.createdById,
      );
      this.nodeMailerService.paymentLinkAfterApprovalUser(
        result.email,
        result.createdById,
      );
      return status;
    }
    if (dto.status == MemberStatus.DEACTIVE) {
      this.nodeMailerService.userAccountReject(
        result.email,
        result.createdById,
      );
      return status;
    }
    return status;
  }
}
