import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  Res,
} from '@nestjs/common';
import { UserLogService } from './user-log.service';
import { CreateUserLogDto, LogPaginationDto } from './dto/create-user-log.dto';
import { UpdateUserLogDto } from './dto/update-user-log.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { Response } from 'express';

@Controller('user-log')
export class UserLogController {
  constructor(private readonly userLogService: UserLogService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'user_log'])
  create(@Body() dto: CreateUserLogDto, @CurrentUser() user: Account) {
    return this.userLogService.create(dto, user.id);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_log'])
  findAll(@Query() dto: LogPaginationDto) {
    return this.userLogService.findAll(dto);
  }

  @Get('download-memberlog-csv')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_log'])
  async downloadMemberLogCSV(
    @Query() dto: LogPaginationDto,
    @Res() res: Response,
  ) {
    const csvFile = await this.userLogService.downloadMemberLogCSV(dto);

    res.header('Content-Type', 'text/csv');
    res.attachment('member-log-list.csv');
    res.send(csvFile);
  }

  @Patch('exit-member')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'user_log'])
  update(@CurrentUser() user: Account, @Body() dto: UpdateUserLogDto) {
    return this.userLogService.update(user.id, dto);
  }
}
