import fs from 'fs-extra';
import PDFDocument from 'pdfkit';
import QRCode from 'qrcode';
import { BillTo } from 'src/enum';

const titleFontSize = 25;
const subTitleFontSize = 15;
const leadingFontSize = 13;
const contentFontSize = 11;
const paperSizeHeight = 841.89;
const paperSizeWidth = 595.28;
const paperMargin = 40;
const lineSpacing = 18;
const bottomGap = 25;

export async function createInvoice(invoice) {
  let doc = new PDFDocument({
    size: [paperSizeWidth, paperSizeHeight],
    margin: paperMargin,
  });

  const fileDir = './uploads/qr';
  const fileName = `${fileDir}/${Date.now().toString()}.png`;
  await fs.ensureDir(fileDir);
  await QRCode.toFile(fileName, 'Thanks For Purchasing Membership');

  generateHeader(doc, fileName, invoice, 50);
  await generateOrderInformation(doc, invoice, 200);
  generateInvoiceTable(doc, invoice, 340);
  generateFooter(doc);

  await fs.unlink(fileName);
  return doc;
}

function generateHeader(doc, fileName, invoice, y) {  
  const qrCodeSize = 100;
  const rightColumnX = paperSizeWidth - paperMargin - qrCodeSize;
  
  doc
    .fontSize(titleFontSize)
    .font('Helvetica-Bold')
    .text('Invoice', { align: 'center' });
  
  doc
    .image(fileName, rightColumnX, y, { width: qrCodeSize, height: qrCodeSize });
  
    if (invoice.userDetail.billTo == BillTo.SELF) {
      doc
        .fontSize(leadingFontSize)
        .font('Helvetica-Bold')
        .text('Member Information:')
        .fontSize(contentFontSize)
        .font('Helvetica')
        .text(invoice.userDetail.name)
        .text(invoice.userDetail.memberId)
        .text(invoice.userDetail.address)
        .text(`Valid From: ${invoice.userDetail.from} To: ${invoice.userDetail.to}`);      
    } else {
      doc
        .fontSize(leadingFontSize)
        .font('Helvetica-Bold')
        .text('Member Information:')
        .fontSize(contentFontSize)
        .font('Helvetica')
        .text(invoice.userDetail.businessName)
        .text(`GST No.- ${invoice.userDetail.gstNumber}`)
        .text(invoice.userDetail.memberId)
        .text(invoice.userDetail.businessAddress1)
        .text(`Valid From: ${invoice.userDetail.from} To: ${invoice.userDetail.to}`);  
    }
}

async function generateOrderInformation(doc, invoice, y) {
  const rightColumnX = paperSizeWidth - paperMargin - 150;
  
  doc
    .fontSize(leadingFontSize)
    .font('Helvetica-Bold')
    .text('Billing Details:', paperMargin, y)
    .moveDown(0.5)
    .fontSize(contentFontSize)
    .font('Helvetica')
    .text(`Name: ${invoice.business.businessName}`)
    .text(`GST No: ${invoice.business.gstNo}`)
    .text(`Address: ${invoice.business.address}`);
  
  doc
    .fontSize(contentFontSize)
    .font('Helvetica-Bold')
    .text(`Invoice No: ${invoice.invoiceNumber}`, rightColumnX, y)
    .text(`Transaction ID: ${invoice.transactionId}`, rightColumnX, y + lineSpacing)
    .text(`Date: ${invoice.paymentDate}`, rightColumnX, y + lineSpacing * 2)
    .text(`Mode: ${invoice.mode}`, rightColumnX, y + lineSpacing * 3);
}

function generateInvoiceTable(doc, invoice, y) {
  doc.font('Helvetica-Bold');
  generateTableRow(doc, y, 'Item', 'Duration(days)', 'GST', 'Amount', true);
  generateHr(doc, y + lineSpacing);
  doc.font('Helvetica');

  let position = y + 30;
  generateTableRow(doc, position, invoice.membershipCardName, invoice.duration, '', invoice.amount.toFixed(2));
  generateHr(doc, position + 18);

  let taxPosition = position + 30;
  if (!invoice.tax || invoice.tax.length === 0) {
    generateTableRow(doc, taxPosition, '', '', `GST: 18%`, ((invoice.amount * 18) / 100).toFixed(2));
    taxPosition += lineSpacing;
  } else {
    invoice.tax.forEach((tax, index) => {
      generateTableRow(doc, taxPosition + index * lineSpacing, '', '', `${tax.taxName}: ${tax.rate}`, ((invoice.amount * parseInt(tax.rate)) / 100).toFixed(2));
    });
    taxPosition += invoice.tax.length * lineSpacing;
  }

  let totalPosition = taxPosition + invoice.tax.length * lineSpacing;
  generateTableTotal(doc, totalPosition, 'TOTAL', formatCurrency(invoice.total), true);
}

function generateTableRow(doc, y, item, duration, gst, amount, isHeader = false) {
  const fontStyle = isHeader ? 'Helvetica-Bold' : 'Helvetica';
  doc
    .fontSize(contentFontSize)
    .font(fontStyle)
    .text(item, paperMargin, y, { width: 180 }) // Adjust width
    .text(duration, paperMargin + 200, y, { width: 80, align: 'center' })
    .text(gst, paperMargin + 280, y, { width: 100, align: 'center' })
    .text(amount, paperMargin + 400, y, { width: 80, align: 'right' });
}

function generateTableTotal(doc, y, title, total, bold = false) {
  const fontStyle = bold ? 'Helvetica-Bold' : 'Helvetica';

  if (title === 'TOTAL') {
    generateHr(doc, y - 1);
    y += 9;
    doc
      .fontSize(contentFontSize)
      .font(fontStyle)
      .text(title, paperMargin, y, { width: 250 }) // Align with "Item"
      .text(total, paperMargin + 400, y, { width: 80, align: 'right' });
  } else {
    doc
      .fontSize(contentFontSize)
      .font(fontStyle)
      .text(title, paperMargin + 300, y, { width: 50, align: 'center' }) // Align with "Duration"
      .text(total, paperMargin + 400, y, { width: 80, align: 'right' });
  }
}

function generateHr(doc, y) {
  doc
    .strokeColor('#aaaaaa')
    .lineWidth(1)
    .moveTo(paperMargin, y)
    .lineTo(paperSizeWidth - paperMargin, y)
    .stroke();
}

function generateFooter(doc) {
  const footerText = 'This is a computer-generated invoice and requires no signature or stamp.';
  const footerY = doc.page.height - doc.page.margins.bottom - 30;
  doc.fontSize(10).text(footerText, paperMargin, footerY, { width: paperSizeWidth - 2 * paperMargin, align: 'center' });
}

function formatCurrency(amount) {
  return parseFloat(amount).toFixed(2);
}
