import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  Req,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { AccountService } from './account.service';
import { Account } from './entities/account.entity';
import { CreateAccountDto, PaginationDto } from './dto/account.dto';
import { DefaultStatus, PermissionAction, UserRole } from '../enum';
import { DefaultStatusDto } from '../common/dto/default-status.dto';
import { AdminCreateUserDto } from './dto/admin-create-user.dto';
import { CreateUserByAdminDto } from './dto/create-user-by-admin.dto';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { MenusService } from 'src/menus/menus.service';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';


@Controller('account')
export class AccountController {
  constructor(
    private readonly accountService: AccountService,
      private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
    
  ) { }
@Post('add-staff')
  @UseGuards(AuthGuard('jwt'), RolesGuard, )
  @Roles(UserRole.ADMIN)
  @CheckPermissions([PermissionAction.CREATE, 'account'])
  async create(@Body() dto: CreateAccountDto, @CurrentUser() user: Account) {
    const account = await this.accountService.create(dto, user.id);
    const menus = await this.menuService.findAll();
    const perms = await this.permissionService.findAll();
    const obj = [];
    menus.forEach((menu) => {
      perms.forEach((perm) => {
        obj.push({
          accountId: account.id,
          menuId: menu.id,
          permissionId: perm.id,
        });
      });
    });
    await this.userPermService.create(obj);
    return account;
  }
  
  @Post('admin/create-user')
   @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
   @CheckPermissions([PermissionAction.CREATE, 'account'])
  async createUserMinimal(
    @Body() dto: CreateUserByAdminDto,
    @CurrentUser() admin: Account
  ) {
    return this.accountService.createUserByAdmin(dto, admin.id);
  }

    @Get('all/users')
  @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
   @CheckPermissions([PermissionAction.READ, 'account'])
  async getAllUsers(@Query() dto: PaginationDto) {
    return this.accountService.getAllUsers(dto);
  }

  @Get('users')
  @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  async getUsers(@Query() dto: PaginationDto) {
    return this.accountService.getUsers(dto);
  }

  @Get('retailers')
   @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
   @CheckPermissions([PermissionAction.READ, 'account'])
  async getRetailers(@Query() dto: PaginationDto) {
    return this.accountService.getRetailer(dto);
  }

  @Get('profile/:id')
   @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'account'])
  async getProfile(@Param('id') id: string) {
    return this.accountService.getProfile(id);
  }
  @Get('all/staff')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  async getStaff(@Query() dto: PaginationDto) {
    return this.accountService.getAllStaff(dto);
  }
  @Get('staff/profile/:id')
   @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(UserRole.ADMIN,)
  async getstaffProfile(@Param('id') id: string) {
    return this.accountService.getstaffProfile(id);
  }

  @Patch('users/status/:id')
 @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'account'])
  async updateStatus(
    @Param('id') id: string,
    @Body() status: DefaultStatusDto) {
    return this.accountService.updateStatus(id, status);
  }
  @Patch('staff/status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  async updateStaffStatus(
    @Param('id') id: string,
    @Body() status: DefaultStatusDto) {
    return this.accountService.updateSatffStatus(id, status);
  }

  // @Patch('status-retailer/:id')
  // @UseGuards(AuthGuard('jwt'), RolesGuard,)
  // @Roles(UserRole.ADMIN)
  // updateStatusRetailrt(@Param('id') id: string, @Body() statusDto: DefaultStatusDto) {
  //   return this.accountService.UpdatestatusRetailer(id, statusDto);
  // }



  @Delete(':id')
   @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'account'])
  async deleteAccount(
    @Param('id') id: string,
    @CurrentUser() currentUser: Account
  ) {
    return this.accountService.deleteAccount(id, currentUser);
  }
    @Delete('staff/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  async deletesatffAccount(
    @Param('id') id: string,
    @CurrentUser() currentUser: Account
  ) {
    return this.accountService.deletestaffAccount(id, currentUser);
  }
}
