import { Injectable, NotFoundException, ForbiddenException, BadRequestException, ConflictException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { Account } from './entities/account.entity';
import { CreateAccountDto, PaginationDto } from './dto/account.dto';
import { DefaultStatus, NotificationType, UserRole } from '../enum';
import { DefaultStatusDto } from '../common/dto/default-status.dto';
import { NotificationsService } from '../notifications/notifications.service';
import { UserDetail } from '../user-details/entities/user-detail.entity';
import { AdminCreateUserDto } from './dto/admin-create-user.dto';
import { CreateUserByAdminDto } from './dto/create-user-by-admin.dto';
import { StaffDetail } from 'src/staff_detail/entities/staff_detail.entity';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import bcrypt from 'bcrypt';
@Injectable()
export class AccountService {
  constructor(
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserDetail) private readonly userDetailRepo: Repository<UserDetail>,
    private readonly notificationsService: NotificationsService,
    @InjectRepository(StaffDetail) private readonly staffRepo: Repository<StaffDetail>,
    private readonly nodeMailerService: NodeMailerService
  ) { }


  async create(dto: CreateAccountDto, createdBy: string) {
    const user = await this.repo.findOne({
      where: { phoneNumber: dto.phoneNumber, roles: UserRole.STAFF },
    });
    if (user) {
      throw new ConflictException('Login id already exists!');
    }
    const encryptedPassword = await bcrypt.hash(dto.password, 13);
    const obj = Object.assign({
      phoneNumber: dto.phoneNumber,

      password: encryptedPassword,
      createdBy,
      roles: UserRole.STAFF,
    });
    const payload = await this.repo.save(obj);
    const object = Object.assign({
      email: dto.email,
      name: dto.name,
      address: dto.address,
      accountId: payload.id,
    });
    await this.staffRepo.save(object);
    return payload;
  }

  async createUserByAdmin(dto: CreateUserByAdminDto, adminId: string) {
    const existingUser = await this.repo.findOne({
      where: { phoneNumber: dto.phoneNumber }
    });

    if (existingUser) {
      throw new BadRequestException(`User already exists with phone number ${dto.phoneNumber}`);
    }
    if (dto.role === UserRole.RETAILER && !dto.businessName) {
      throw new BadRequestException('Business name is required for retailers');
    }
    const account = await this.repo.save({
      phoneNumber: dto.phoneNumber,
      roles: dto.role,
      status: dto.status || DefaultStatus.ACTIVE,
      createdBy: adminId
    });
    if (dto.name || dto.email || dto.address || dto.businessName || dto.alternativePhoneNumber) {
      const userDetailData: any = {
        accountId: account.id
      };
      if (dto.name) userDetailData.name = dto.name;
      if (dto.email) userDetailData.email = dto.email;
      if (dto.address) userDetailData.address = dto.address;
      if (dto.businessName) userDetailData.businessName = dto.businessName;
      if (dto.alternativePhoneNumber) userDetailData.alternativePhoneNumber = dto.alternativePhoneNumber;

      await this.userDetailRepo.save(userDetailData);
    }
    await this.notificationsService.create({
      title: 'Account Created',
      desc: `Your ${dto.role.toLowerCase()} account has been created by admin. Welcome to the platform!`,
      accountId: account.id,
      type: NotificationType.ACCOUNT
    });
    await this.notifyAdmins('New User Created', `Admin created ${dto.role.toLowerCase()} account for ${dto.name || 'user'} (${dto.phoneNumber})`);

    return {
      message: `${dto.role} account created successfully`,
      accountId: account.id,
    };
  }

  async getAllUsers(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'userDetail.id',
        'userDetail.name',
        'userDetail.email',
        'userDetail.address',
      ])
     .where('account.roles IN (:...roles)', { roles: [UserRole.USER, UserRole.RETAILER] })
      .andWhere('account.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere(
        new Brackets((qb) => {
          if (dto.keyword) {
            qb.where('userDetail.name LIKE :keyword', { keyword: `%${keyword}%` })
              .orWhere('userDetail.email LIKE :keyword', { keyword: `%${keyword}%` })
              .orWhere('account.phoneNumber LIKE :keyword', { keyword: `%${keyword}%` });
          }
        }),
      );
    const [result, total] = await query
      .orderBy('account.createdAt', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }
  async getUsers(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'userDetail.id',
        'userDetail.name',
        'userDetail.email',
        'userDetail.address',
      ])
      .where('account.roles = :role', { role: UserRole.USER })
      .andWhere(
        new Brackets((qb) => {
          if (dto.keyword) {
            qb.where('userDetail.name   LIKE :keyword', { keyword: `%${keyword}%` })
              .orWhere('userDetail.email  LIKE :keyword', { keyword: `%${keyword}%` });
          }
        }),
      );
    if (dto.status) {
      query.andWhere('account.status = :status', { status: dto.status });
    }
    const [result, total] = await query
      .orderBy('account.createdAt', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async getRetailer(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'userDetail.id',
        'userDetail.name',
        'userDetail.email',
        'userDetail.address',
        'userDetail.businessName',
        'userDetail.documentUrl',
      ])
      .where('account.roles = :role', { role: UserRole.RETAILER })
      .andWhere(
        new Brackets((qb) => {
          if (dto.keyword) {
            qb.where('userDetail.name   LIKE :keyword', { keyword: `%${keyword}%` })
              .orWhere('userDetail.email  LIKE :keyword', { keyword: `%${keyword}%` });
          }
        }),
      );
    if (dto.status) {
      query.andWhere('account.status = :status', { status: dto.status });
    }
    const [result, total] = await query
      .orderBy('account.createdAt', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async getProfile(id: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'userDetail.id',
        'userDetail.name',
        'userDetail.email',
        'userDetail.profile',
      ])
      .where('account.id = :id', { id })
      .getOne();

    if (!result) {
      throw new NotFoundException('Profile not found');
    }

    return result;
  }
  async updateStatus(id: string, dto: DefaultStatusDto) {
    const account = await this.repo.findOne({ where: { id } });
    const user = await this.userDetailRepo.findOne({
      where: { accountId: id }
    });

    if (!account) {
      throw new NotFoundException('Account not found');
    }
    const previousStatus = account.status;
    const obj = Object.assign(account, dto);
    const savedAccount = await this.repo.save(obj);
    if (account.roles === UserRole.RETAILER && previousStatus !== dto.status) {
      let title = 'Account Status Updated';
      let message = `Your account status has been updated to ${dto.status}.`;

      if (previousStatus === DefaultStatus.PENDING && dto.status === DefaultStatus.ACTIVE) {
        title = 'Account Approved';
        message = 'Your retailer account has been approved. You can now log in and start using the platform.';
      } else if (dto.status === DefaultStatus.SUSPENDED) {
        title = 'Account Suspended';
        message = 'Your account has been suspended. Please contact support for more information.';
      }

      await this.notificationsService.create({
        title,
        desc: message,
        accountId: account.id,
        type: NotificationType.ACCOUNT_STATUS_CHANGE
      });
      
      try {
        await this.nodeMailerService.sendCustomNotification(
          user.email,
          user.name,
          title,
          message
        );
      } catch (error) {
        console.error('Failed to send status update email:', error);
      }

    }
    return savedAccount;
  }

  async updateSatffStatus(id: string, status: DefaultStatusDto) {
    const account = await this.repo.findOne({ where: { id } });
    if (!account) {
      throw new NotFoundException('Account not found');
    }
    const obj = Object.assign(account, status);
    const result = await this.repo.save(account);

    await this.notifyAdmins('Staff Status Updated', `Admin updated user status to ${status.status} for account: ${account.phoneNumber}`);

    return result;
  }
  async getstaffProfile(id: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'staffDetail.id',
        'staffDetail.name',
        'staffDetail.email',
      ])
      .where('account.id = :id', { id })
      .getOne();

    if (!result) {
      throw new NotFoundException('Profile not found');
    }

    return result;
  }



  async UpdatestatusRetailer(id: string, dto: DefaultStatusDto) {
    const account = await this.repo.findOne({
      where: { id: id },
      relations: ['userDetail']
    });
    const user = await this.userDetailRepo.findOne({
      where: { accountId: id }
    });

    if (!account) {
      throw new NotFoundException('Account not found!');
    }
    const previousStatus = account.status;
    const obj = Object.assign(account, dto);
    const savedAccount = await this.repo.save(obj);

    if (account.roles === UserRole.RETAILER && previousStatus !== dto.status) {
      let title = 'Account Status Updated';
      let message = `Your account status has been updated to ${dto.status}.`;

      if (previousStatus === DefaultStatus.PENDING && dto.status === DefaultStatus.ACTIVE) {
        title = 'Account Approved';
        message = 'Your retailer account has been approved. You can now log in and start using the platform.';
      } else if (dto.status === DefaultStatus.SUSPENDED) {
        title = 'Account Suspended';
        message = 'Your account has been suspended. Please contact support for more information.';
      }

      await this.notificationsService.create({
        title,
        desc: message,
        accountId: account.id,
        type: NotificationType.ACCOUNT_STATUS_CHANGE
      });
      
      try {
        await this.nodeMailerService.sendCustomNotification(
          user.email,
          user.name,
          title,
          message
        );
      } catch (error) {
        console.error('Failed to send status update email:', error);
      }

    }
    return savedAccount;
  }

  async getAllStaff(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',
        'staffDetail.id',
        'staffDetail.name',
        'staffDetail.email',
        'staffDetail.address'
      ])
      .where('account.roles = :role', { role: UserRole.STAFF })
      .andWhere(
        new Brackets((qb) => {
          if (dto.keyword) {
            qb.where('staffDetail.name   LIKE :keyword', { keyword: `%${keyword}%` })
              .orWhere('staffDetail.email  LIKE :keyword', { keyword: `%${keyword}%` });
          }
        }),
      );
    if (dto.status) {
      query.andWhere('account.status = :status', { status: dto.status });
    }
    const [result, total] = await query
      .orderBy('account.createdAt', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  private async notifyAdmins(title: string, desc: string) {
    const admins = await this.repo.find({ where: { roles: UserRole.ADMIN } });
    for (const admin of admins) {
      await this.notificationsService.create({
        title,
        desc,
        accountId: admin.id,
        type: NotificationType.SYSTEM,
      });
    }
  }

  async deleteAccount(id: string, currentUser: Account) {
    if (currentUser.roles !== UserRole.ADMIN) {
      throw new ForbiddenException('Only administrators can delete accounts');
    }

    const account = await this.repo.findOne({
      where: { id }
    });

    if (!account) {
      throw new NotFoundException('Account not found');
    }

    const userDetail = await this.userDetailRepo.findOne({ where: { accountId: id } });
    const userName = userDetail?.name || 'Unknown';

    await this.notificationsService.clearAllNotifications(id);
    await this.repo.remove(account);

    await this.notifyAdmins('Account Deleted', `Admin deleted ${account.roles.toLowerCase()} account: ${userName} (${account.phoneNumber})`);

    return { message: 'Account and related user data deleted successfully' };
  }

  async deletestaffAccount(id: string, currentUser: Account) {
    if (currentUser.roles !== UserRole.ADMIN) {
      throw new ForbiddenException('Only administrators can delete accounts');
    }

    const account = await this.repo.findOne({
      where: { id }
    });

    if (!account) {
      throw new NotFoundException('Account not found');
    }

    const staffDetail = await this.staffRepo.findOne({ where: { accountId: id } });
    const staffName = staffDetail?.name || 'Unknown';

    await this.notificationsService.clearAllNotifications(id);
    await this.repo.remove(account);

    await this.notifyAdmins('Account Deleted', `Admin deleted ${account.roles.toLowerCase()} account: ${staffName} (${account.phoneNumber})`);

    return { message: 'Account and related Staff data deleted successfully' };
  }
}