import {
  Body,
  Controller,
  FileTypeValidator,
  Get,
  Ip,
  MaxFileSizeValidator,
  Param,
  ParseFilePipe,
  Post,
  Req,
  UploadedFile,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { AuthService } from './auth.service';
import { UserRole } from 'src/enum';
import { RegisterRetailerDto, RegisterUserDto, SendOtpDto, VerifyOtpDto } from './dto/reg.dto';
import { FileInterceptor } from '@nestjs/platform-express';
import { AdminSigninDto, ForgotPassDto, LoginRequestDto, VerifyLoginOtpDto } from './dto/login.dto';
import { diskStorage } from 'multer';
import { extname } from 'path';

@Controller('auth')
export class AuthController {
  constructor(private readonly authService: AuthService) { }

  @Post('admin/login')
  signin(@Body() dto: AdminSigninDto) {
    return this.authService.signIn(dto.loginId, dto.password);
  }

  @Post('send-otp')
  sendOtp(@Body() phone: SendOtpDto) {
    return this.authService.sendOtp(phone);
  }

  @Post('verify-otp')
  verifyOtp(@Body() dto: VerifyOtpDto) {
    return this.authService.verifyOtp(dto);
  }

  @Post('register-user')
  register(@Body() dto: RegisterUserDto) {
    return this.authService.registerUser(dto);
  }
  @Post('forgotPass')
  adminStaffForgotPass(@Body() dto: ForgotPassDto) {
    return this.authService.forgotPass(dto);
  }


  @Post('register-retailer')
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/UserDetail/doc',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async registerRetailer(
    @Body() dto: RegisterRetailerDto,
    @UploadedFile(
      new ParseFilePipe({
        fileIsRequired: true,
        validators: [
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 5 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.authService.registerRetailer(dto, file.path);
  }


  //LogIn
  @Post('login/send-otp')
  loginSendOtp(@Body() dto: LoginRequestDto) {
    return this.authService.loginSendOtp(dto.phoneNumber, dto.role);
  }

  @Post('login/verify')
  verifyLoginOtp(@Body() dto: VerifyLoginOtpDto) {
    return this.authService.verifyLoginOtp(dto.phoneNumber, dto.otp, dto.role);
  }
  @Post('otp/verify')
  adminStaffVerifyOtp(@Body() dto: VerifyOtpDto) {
    return this.authService.adminStaffVerifyOtp(dto.phoneNumber, dto.otp);
  }


  @Post('resetPass')
  adminStaffResetPassword(@Body() dto: ForgotPassDto) {
    return this.authService.resetPassword(dto);
  }
}
