import { Module, forwardRef } from '@nestjs/common';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { JwtModule } from '@nestjs/jwt';
import { PassportModule } from '@nestjs/passport';
import { TypeOrmModule } from '@nestjs/typeorm';

import { Account } from 'src/account/entities/account.entity';
import { AuthController } from './auth.controller';
import { AuthService } from './auth.service';
import { HttpModule } from '@nestjs/axios';
import { JwtStrategy } from './strategy/jwt.strategy';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { UserDetailsService } from '../user-details/user-details.service';
import { Notification } from '../notifications/entities/notification.entity';
import { NotificationsModule } from '../notifications/notifications.module';
import { NodeMailerModule } from 'src/node-mailer/node-mailer.module';
import { CaslAbilityFactory } from './factory/casl-ability.factory';
import { PermissionsGuard } from './guards/permissions.guard';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';

@Module({
  imports: [
    TypeOrmModule.forFeature([
      Account,
      UserDetail,
       UserPermission,
    ]),
    PassportModule.register({ defaultStrategy: 'jwt' }),
    JwtModule.registerAsync({
      imports: [ConfigModule],
      inject: [ConfigService],
      useFactory: () => {
        return {
          secret: process.env.RN_JWT_SECRET,
          signOptions: {
            expiresIn: process.env.RN_JWT_EXPIRE,
          },
        };
      },
    }),
    HttpModule.register({
      timeout: 5000,
      maxRedirects: 5,
    }),
    forwardRef(() => NotificationsModule),
    NodeMailerModule
  ],
  controllers: [AuthController],
  providers: [AuthService, JwtStrategy, UserDetailsService,CaslAbilityFactory, PermissionsGuard],
  exports: [
    AuthService,
    JwtStrategy,
    PassportModule,
    JwtModule,
    CaslAbilityFactory,
    PermissionsGuard,
  ],
})
export class AuthModule { }
