import { HttpService } from '@nestjs/axios';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { Cache } from 'cache-manager';
import { Account } from 'src/account/entities/account.entity';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import APIFeatures from 'src/utils/apiFeatures.utils';
import { Repository } from 'typeorm';
import { DefaultStatus, NotificationType, UserRole } from '../enum';
import { RegisterRetailerDto, RegisterUserDto, SendOtpDto, VerifyOtpDto } from './dto/reg.dto';
import { Roles } from './decorators/roles.decorator';
import { stat } from 'fs';
import { UserDetailsService } from '../user-details/user-details.service';
import { NotificationsService } from '../notifications/notifications.service';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { log } from 'console';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';
import { ForgotPassDto } from './dto/login.dto';
import { sendOtpReg } from 'src/utils/sms.utils';


@Injectable()
export class AuthService {
  constructor(
    private jwtService: JwtService,
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserPermission)
    private readonly upRepo: Repository<UserPermission>,
    @InjectRepository(UserDetail)
    private readonly userDetailRepo: Repository<UserDetail>,
    private readonly userDetailsService: UserDetailsService,
    private readonly httpService: HttpService,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    private readonly notificationsService: NotificationsService,
    private readonly nodeMailerService: NodeMailerService,
  ) { }


  //Register
  async sendOtp(dto: SendOtpDto) {
    // const phoneNumber = Number(dto.phoneNumber);
    const existingUser = await this.repo.findOneBy({ phoneNumber: dto.phoneNumber });

    if (existingUser) {
      throw new BadRequestException(
        `User already exists with role ${existingUser.roles}`
      );
    }
   const otp = Math.floor(100000 + Math.random() * 900000).toString();
    //const otp = "743211";
    const sendotp = Number(otp)
    sendOtpReg(dto.phoneNumber, sendotp);

    await this.cacheManager.set(`otp:${dto.phoneNumber}`, otp, 10 * 60 * 1000);
    return { message: 'OTP sent successfully' };
  }

  async verifyOtp(dto: VerifyOtpDto) {
    const sentOtp = await this.cacheManager.get(`otp:${dto.phoneNumber}`);
    if (!sentOtp || sentOtp != dto.otp) {
      throw new UnauthorizedException('Invalid OTP');
    }
    await this.cacheManager.set(`otp_verified:${dto.phoneNumber}`, true, 10 * 60 * 1000);
    return { message: 'OTP verified successfully' };
  }


  async registerUser(dto: RegisterUserDto) {
    const isVerified = await this.cacheManager.get(`otp_verified:${dto.phoneNumber}`);
    if (!isVerified) {
      throw new UnauthorizedException('Please verify your phone number first');
    }

    const existing = await this.repo.findOneBy({ phoneNumber: dto.phoneNumber, roles: UserRole.USER || UserRole.RETAILER });
    if (existing) {
      throw new BadRequestException('User already exists');
    }

    const user = Object.create({
      phoneNumber: dto.phoneNumber,
      roles: UserRole.USER,
      status: 'ACTIVE',
    });
    const saveUser = await this.repo.save(user);

    await this.userDetailRepo.save({
      accountId: saveUser.id,
      name: dto.name,
      email: dto.email,
      address: dto.address,
    });

    await this.cacheManager.del(`otp_verified:${dto.phoneNumber}`);
    return { message: 'User registered successfully' };
  }

  async registerRetailer(dto: RegisterRetailerDto, file: string) {

    const isVerified = await this.cacheManager.get(`otp_verified:${dto.phoneNumber}`);
    if (!isVerified) {
      throw new UnauthorizedException('Please verify your phone number first');
    }

    const existingAccount = await this.repo.findOne({
      where: { phoneNumber: dto.phoneNumber, roles: UserRole.RETAILER },
    });

    if (existingAccount) {
      throw new ConflictException('Mobile number already registered');
    }

    const account = Object.create({
      phoneNumber: dto.phoneNumber,
      roles: UserRole.RETAILER,
      status: DefaultStatus.PENDING
    });
    const savedAccount = await this.repo.save(account);

    const retailer = Object.create({
      name: dto.name,
      email: dto.email,
      businessName: dto.businessName,
      phoneNumber: dto.phoneNumber,
      address: dto.address,
      accountId: savedAccount.id
    });
    const savedUser = await this.userDetailRepo.save(retailer);
    await this.userDetailsService.docoment(file, savedUser)
    await this.notifyAdmins('New Retailer Registration', `A new retailer ${dto.name} (${dto.businessName}) has registered and is pending approval. Please verify their details and documents.`);
    if (dto.email) {
      await this.nodeMailerService.sendCustomNotification(
        dto.email,
        dto.name,
        'Retailer Registration Successful',
        'Your retailer registration is complete. Please wait 48 hours for admin verification. You will be notified once approved.'
      );
    }
    await this.cacheManager.del(`otp_verified:${dto.phoneNumber}`);
    return { message: 'Retailer registered successfully' };
  }

  async signIn(loginId: string, password: string) {
    const user = await this.getUserDetails(loginId);
    if (!user || !user.password) {
      throw new UnauthorizedException('Invalid Credentials');
    }

    if (![UserRole.ADMIN, UserRole.STAFF].includes(user.roles)) {
      throw new UnauthorizedException('Access denied');
    }

    const comparePassword = await bcrypt.compare(password, user.password);

    if (!comparePassword) {
      throw new UnauthorizedException('Invalid Credentials');
    }

    const token = await APIFeatures.assignJwtToken(user.id, this.jwtService);
    return { token, roles: user.roles };
  }




  // Login USER and Retailer
  async loginSendOtp(phoneNumber: string, role: UserRole) {

    const user = await this.repo.findOneBy({ phoneNumber, roles: role });

    if (!user) {
      throw new UnauthorizedException('Account not found !');
    }
    if (role === UserRole.RETAILER && user.status !== DefaultStatus.ACTIVE) {
      throw new UnauthorizedException('Your account is not active. Please contact admin.');
    }
     const otp = Math.floor(100000 + Math.random() * 900000).toString();
    //const otp = "783211";
    const sendotp = Number(otp)
    sendOtpReg(phoneNumber, sendotp);
    await this.cacheManager.set(`login_otp_${phoneNumber}_${role}`, otp, 10 * 60 * 1000);
    return { message: 'OTP sent successfully' };
  }

  async verifyLoginOtp(phoneNumber: string, otp: string, role: UserRole) {
    const cachedOtp = await this.cacheManager.get(`login_otp_${phoneNumber}_${role}`);
    if (!cachedOtp || String(cachedOtp) !== otp) {
      throw new UnauthorizedException('Invalid OTP');
    }

    const user = await this.repo.findOne({
      where: { phoneNumber, roles: role },
      relations: ['userDetail'],
    });

    if (!user) throw new UnauthorizedException(`${role} account not found`);

    if (role === UserRole.RETAILER && user.status === DefaultStatus.PENDING) {
      throw new UnauthorizedException('Retailer not approved by admin');
    }

    const token = await APIFeatures.assignJwtToken(user.id, this.jwtService);

    await this.cacheManager.del(`login_otp_${phoneNumber}_${role}`);

    return {
      token,
      role: user.roles,
      status: user.status,
    };
  }

  async forgotPass(dto: ForgotPassDto) {
    const user = await this.repo
      .createQueryBuilder('account')
      .where(
        'account.phoneNumber = :phoneNumber AND (account.roles = :staffRole OR account.roles = :adminRole) AND account.status = :status',
        {
          phoneNumber: dto.phoneNumber,
          staffRole: UserRole.STAFF,
          adminRole: UserRole.ADMIN,
          status: DefaultStatus.ACTIVE,
        },
      )
      .getOne();
    if (!user) {
      throw new NotFoundException(
        'Phone number does not exist.',
      );
    }
    
    
 const otp = Math.floor(100000 + Math.random() * 900000).toString();
    //const otp = "783211";
    const sendotp = Number(otp)
    sendOtpReg(dto.phoneNumber, sendotp);

    await this.cacheManager.set(dto.phoneNumber, otp, 15 * 60 * 1000);
    
    return { message: 'OTP sent to your phone number' };
  }

  async adminStaffVerifyOtp(phoneNumber: string, otp: string) {
    const storedOtp = await this.cacheManager.get<string>(phoneNumber);
    if (storedOtp !== otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    return { message: 'OTP Macthed.' };
  }

  async resetPassword(dto: ForgotPassDto) {
    const user = await this.repo
      .createQueryBuilder('account')
      .where(
        'account.phoneNumber = :phoneNumber AND (account.roles = :staffRole OR account.roles = :adminRole) AND account.status = :status',
        {
          phoneNumber: dto.phoneNumber,
          staffRole: UserRole.STAFF,
          adminRole: UserRole.ADMIN,
          status: DefaultStatus.ACTIVE,
        },
      )
      .getOne();
    if (!user) {
      throw new NotFoundException(
        'Phone number does not exist . Please check your phone number.',
      );
    }
    const hashedPassword = await bcrypt.hash(dto.newPassword, 10);
    user.password = hashedPassword;

    await this.repo.save(user);
    await this.cacheManager.del(dto.phoneNumber);

    return { message: 'Password reset successfully' };
  }

  validate(id: string) {
    return this.getUserDetails(id);
  }

  private getUserDetails = async (
    id: string,
    role?: UserRole,
  ): Promise<any> => {
    // let result = await this.cacheManager.get('userDetail' + id);
    // if (!result) {
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdBy',
        'userDetail.id',
        'userDetail.name',
      ]);
    if (role === UserRole.USER) {
      query.andWhere('account.roles = :roles', { roles: UserRole.USER });
    } else if (role === UserRole.RETAILER) {
      query.andWhere('account.roles = :roles', { roles: UserRole.RETAILER });
    } else if (role === UserRole.ADMIN) {
      query.andWhere('account.roles = :roles', { roles: UserRole.ADMIN });
    }
    const result = await query
      .andWhere('account.id = :id OR account.phoneNumber = :phoneNumber', {
        id: id,
        phoneNumber: id,
      })
      .getOne();
    // this.cacheManager.set('userDetail' + id, result, 7 * 24 * 60 * 60 * 1000);
    // }
    if (!result) {
      throw new UnauthorizedException('Account not found!');
    }
    return result;
  };



  private async notifyAdmins(title: string, desc: string) {
    const admins = await this.repo.find({ where: { roles: UserRole.ADMIN } });
    for (const admin of admins) {
      await this.notificationsService.create({
        title,
        desc,
        accountId: admin.id,
        type: NotificationType.SYSTEM,
      });
    }

    try {
      await this.nodeMailerService.sendCustomNotificationAdmin(title, desc);
    } catch (error) {
      console.error('Failed to send admin notification email:', error);
    }
  }

  findPermission(accountId: string) {
    return this.getPermissions(accountId);
  }

  private getPermissions = async (accountId: string): Promise<any> => {
    let result = await this.cacheManager.get('userPermission' + accountId);
    if (!result) {
      result = await this.upRepo.find({
        relations: ['permission', 'menu'],
        where: { accountId, status: true },
      });
      this.cacheManager.set(
        'userPermission' + accountId,
        result,
        7 * 24 * 60 * 60 * 1000,
      );
    }
    return result;
  };
}



