import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Query,
  UseGuards,
  UsePipes,
  ValidationPipe,
  UploadedFile,
  ParseFilePipe,
  FileTypeValidator,
  MaxFileSizeValidator,
  UseInterceptors,
  Put,
} from '@nestjs/common';

import { CreateBrandDto, PaginationDto, } from './dto/create-brand.dto';
import { UpdateBrandDto } from './dto/update-brand.dto';
import { RolesGuard } from '../auth/guards/roles.guard';
import { Roles } from '../auth/decorators/roles.decorator';
import { PermissionAction, UserRole } from '../enum';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { Category } from '../category/entities/category.entity';
import { Product } from '../product/entities/product.entity';
import { BrandService } from './brand.service';
import { CurrentUser } from '../auth/decorators/current-user.decorator';
import { Account } from '../account/entities/account.entity';
import { extname } from 'path';
import { diskStorage } from 'multer';
import { FileInterceptor } from '@nestjs/platform-express';
import { AuthGuard } from '@nestjs/passport';
import { StatusDto } from '../category/dto/category.dto';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';

@Controller('brands')
export class BrandController {
  constructor(private readonly brandService: BrandService) { }

  @Post()
   @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
   @CheckPermissions([PermissionAction.CREATE, 'brand'])
  create(@Body() createBrandDto: CreateBrandDto) {
    return this.brandService.create(createBrandDto);
  }

  @Get('all')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'brand'])
  findAll(@Query() dto: PaginationDto) {
    return this.brandService.findAll(dto);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  findAllbyUser(@Query() dto: PaginationDto) {
    return this.brandService.findAllByUser(dto);
  }

  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.brandService.findOne(id);
  }

  @Put('image/:id')
   @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'brand'])
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/brand',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async image(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          // new FileTypeValidator({
          //   fileType: /^image\/(png|jpe?g)$/,
          // }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,

  ) {
    const fileData = await this.brandService.findOne(id);
    return this.brandService.image(file.path, fileData);
  }

  @Patch('status/:id')
   @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'brand'])
  updateStatus(@Param('id') id: string, @Body() statusDto: StatusDto) {
    return this.brandService.status(id, statusDto);
  }


  @Patch(':id')
 @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'brand'])
  update(@Param('id') id: string, @Body() updateBrandDto: UpdateBrandDto) {
    return this.brandService.update(id, updateBrandDto);
  }

  @Delete(':id')
 @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'brand'])
  remove(@Param('id') id: string) {
    return this.brandService.remove(id);
  }

}
