import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreateBrandDto, PaginationDto, PaginationWithFilterDto } from './dto/create-brand.dto';
import { UpdateBrandDto } from './dto/update-brand.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Brand } from './entities/brand.entity';
import { Repository, In, Brackets } from 'typeorm';
import { Product } from '../product/entities/product.entity';
import { Category } from '../category/entities/category.entity';
import { Wishlist } from '../wishlist/entities/wishlist.entity';
import { StatusDto } from '../category/dto/category.dto';
import { DefaultStatus } from 'src/enum';

@Injectable()
export class BrandService {
  constructor(
    @InjectRepository(Brand) private readonly brandRepository: Repository<Brand>,
  ) { }

  async create(dto: CreateBrandDto) {
    const result = await this.brandRepository.findOne({
      where: { name: dto.name },
    });
    if (result) {
      throw new ConflictException('Brand already exist');
    }
    const obj = Object.create(dto);
    return this.brandRepository.save(obj);
  }

  async findAll(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const queryBuilder = this.brandRepository
      .createQueryBuilder('brand')
      .select([
        'brand.id',
        'brand.name',
        'brand.image',
        'brand.imageName',
        'brand.status',
      ])
      .andWhere(
        new Brackets((qb) => {
          qb.where('brand.name LIKE :name', { name: `%${keyword}%` })
        }),
      );
    if (dto.status) {
      queryBuilder.andWhere('brand.status = :status', { status: dto.status });
    }
    const [result, count] = await queryBuilder
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }

  async findAllByUser(dto: PaginationDto) {
    const keyword = dto.keyword || '';

    const queryBuilder = this.brandRepository
      .createQueryBuilder('brand')
      .leftJoin('brand.products', 'product')
      .leftJoin('product.category', 'category')
      .select([
        'brand.id',
        'brand.name',
        'brand.status',
        'brand.image',
        'brand.imageName',
      ])
      .where('brand.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere(
        new Brackets((qb) => {
          qb.where('brand.name LIKE :name', { name: `%${keyword}%` });
        }),
      );

    if (dto.categoryId) {
      queryBuilder.andWhere('category.id = :categoryId', {
        categoryId: dto.categoryId,
      });
    }

    const [result, count] = await queryBuilder
      .groupBy('brand.id')
      .orderBy('brand.name', 'ASC')
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, count };
  }


  async findOne(id: string) {
    const result = await this.brandRepository.findOne({ where: { id: id, }, });
    if (!result) {
      throw new NotFoundException('Brand not found');
    }
    return result;
  }
  async image(image: string, result: Brand) {
    const obj = Object.assign(result, {
      image: process.env.RN_CDN_LINK + image,
      imageName: image,
    });
    return this.brandRepository.save(obj);
  }

  async status(id: string, dto: StatusDto) {
    const menu = await this.brandRepository.findOne({ where: { id: id } });
    if (!menu) {
      throw new NotFoundException('Brand not found!');
    }
    const obj = Object.assign(menu, dto);
    return this.brandRepository.save(obj);
  }

  async update(id: string, dto: UpdateBrandDto) {
    const result = await this.brandRepository.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Brand not found');
    }
    const obj = Object.assign(result, dto);
    return this.brandRepository.save(obj);
  }

  async remove(id: string) {
    const result = await this.brandRepository.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Brand not found');
    }
    const products = await this.brandRepository
      .createQueryBuilder('brand')
      .leftJoinAndSelect('brand.products', 'product')
      .where('brand.id = :id', { id })
      .getOne();
    if (products.products.length > 0) {
      throw new ConflictException('Brand has products can not be deleted');
    }
    await this.brandRepository.delete(id);
    return { message: 'Brand deleted successfully' };
  }
}
