import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, Query } from '@nestjs/common';
import { CartService } from './cart.service';
import { ApplyCouponDto, CreateCartDto } from './dto/create-cart.dto';
import { UpdateCartDto, UpdateCartItemQtyDto } from './dto/update-cart.dto';
import { AuthGuard } from '@nestjs/passport';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { UserRole } from 'src/enum';

import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';

@Controller('cart')
export class CartController {
  constructor(private readonly cartService: CartService) { }

  @Post('user/add')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async addToCartUser(
    @Query() dto: CreateCartDto,
    @CurrentUser() user: Account,
  ) {
    dto.accountId = user.id;
    return this.cartService.addToCartUser(dto, user.id);
  }
  @Post('retailer/add')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async addToCartRetailer(
    @Query() dto: CreateCartDto,
    @CurrentUser() user: Account,
  ) {
    dto.accountId = user.id;
    return this.cartService.addToCartRetailer(dto, user.id);
  }

  @Patch('apply-coupon')
  applyCoupon(
    @Body() dto: ApplyCouponDto,
    @CurrentUser() user: Account,
  ) {
    return this.cartService.applyCoupon(user.id, dto.couponCode);
  }

  @Patch('item/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  updateCartItemQty(
    @Param('id') cartItemId: string,
    @Body() dto: UpdateCartItemQtyDto,
    @CurrentUser() user: Account,

  ) {
    return this.cartService.updateCartItemQuantity(cartItemId, dto, user.id);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  async getCartItems(@CurrentUser() user: Account) {
    const accountId = user.id;
    return this.cartService.getCartItems(accountId);
  }

  @Delete(':id')
    @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  remove(@Param('id') id: string, @CurrentUser() user: Account) {
    const accountId = user.id;
    return this.cartService.removes(id, accountId);
  }
}
