import { BadRequestException, ForbiddenException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateCartDto } from './dto/create-cart.dto';
import { UpdateCartDto, UpdateCartItemQtyDto } from './dto/update-cart.dto';
import { Cart } from 'src/cart/entities/cart.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CartItem } from 'src/cart-item/entities/cart-item.entity';
import { Account } from './../account/entities/account.entity';
import { Product } from 'src/product/entities/product.entity';
import { UserRole } from 'src/enum';
import { UpdateCartItemDto } from 'src/cart-item/dto/update-cart-item.dto';
import { Coupon } from '../coupon/entities/coupon.entity';
import { UserCoupon } from '../coupon/entities/user-coupon.entity';
import { CouponStatus } from '../enum';
import { NotificationsService } from '../notifications/notifications.service';


@Injectable()
export class CartService {

  constructor(
    @InjectRepository(Cart) private readonly cartRepo: Repository<Cart>,
    @InjectRepository(Product) private readonly productrepo: Repository<Product>,
    @InjectRepository(CartItem) private readonly cartItemrepo: Repository<CartItem>,
    @InjectRepository(Account) private readonly accountRepo: Repository<Account>,
    @InjectRepository(Coupon)
    private readonly couponRepo: Repository<Coupon>,
    @InjectRepository(UserCoupon)
    private readonly userCouponRepo: Repository<UserCoupon>,
    private readonly notificationService: NotificationsService,

  ) { }


  async addToCartUser(dto: CreateCartDto, accountId: string) {
    let cart = await this.cartRepo.findOne({ where: { accountId } });

    if (!cart) {
      cart = await this.cartRepo.save({ accountId, totalPrice: 0 });
    }

    const existingItem = await this.cartItemrepo.findOne({
      where: { cartId: cart.id, productId: dto.productId },
    });

    if (existingItem) {
      throw new BadRequestException('Product already in cart');
    }

    const product = await this.productrepo.findOneBy({ id: dto.productId });
    if (!product) {
      throw new NotFoundException('Product not found');
    }

    if (product.quantity <= 0) {
      throw new BadRequestException('Product is out of stock');
    }

    const account = await this.accountRepo.findOneBy({ id: accountId });
    if (!account) {
      throw new NotFoundException('Account not found');
    }

    const price =product.userPrice;
    const quantity = 1;
    const subTotalPrice = price * quantity;
    const role = account.roles;
    const newItem = Object.create({
      cartId: cart.id,
      productId: dto.productId,
      quantity,
      accountId,
      price,
      subTotalPrice,
      role
    });
    const savedItem = await this.cartItemrepo.save(newItem);
    cart.totalPrice = Number(cart.totalPrice) + subTotalPrice;
    await this.cartRepo.save(cart);
    return {
      message: 'Product added to cart successfully',
      item: savedItem,
    };
  }

    async addToCartRetailer(dto: CreateCartDto, accountId: string) {
    let cart = await this.cartRepo.findOne({ where: { accountId } });
    if (!cart) {
      cart = await this.cartRepo.save({ accountId, totalPrice: 0 });
    }
    const existingItem = await this.cartItemrepo.findOne({
      where: { cartId: cart.id, productId: dto.productId },
    });

    if (existingItem) {
      throw new BadRequestException('Product already in cart');
    }
    const product = await this.productrepo.findOneBy({ id: dto.productId });
    if (!product) {
      throw new NotFoundException('Product not found');
    }
    if (product.quantity <= 0) {
      throw new BadRequestException('Product is out of stock');
    }
    const account = await this.accountRepo.findOneBy({ id: accountId });
    if (!account) {
      throw new NotFoundException('Account not found');
    }
    const price =product.retailerPrice;
    const quantity = 1;
    const subTotalPrice = price * quantity;
    const role = account.roles;

    const newItem = Object.create({
      cartId: cart.id,
      productId: dto.productId,
      quantity,
      accountId,
      price,
      subTotalPrice,
      role
    });
    const savedItem = await this.cartItemrepo.save(newItem);

    cart.totalPrice = Number(cart.totalPrice) + subTotalPrice;
    await this.cartRepo.save(cart);

    return {
      message: 'Product added to cart successfully',
      item: savedItem,
    };
  }

  async applyCoupon(accountId: string, couponCode: string) {
    const cart = await this.cartRepo.findOne({
      where: { accountId },
      relations: ['cartItems', 'coupon'],
    });

    if (!cart) {
      throw new NotFoundException('Cart not found');
    }

    const userCoupon = await this.userCouponRepo.findOne({
      where: { accountId, status: CouponStatus.ACTIVE },
      relations: ['coupon'],
    });

    if (!userCoupon || userCoupon.coupon.code !== couponCode) {
      throw new BadRequestException('Invalid coupon or not assigned to you');
    }

    const coupon = userCoupon.coupon;

    if (!coupon.isActive || (coupon.expiresAt && new Date() > coupon.expiresAt)) {
      throw new BadRequestException('Coupon is expired or inactive');
    }

    if (coupon.minPurchase && cart.totalPrice < coupon.minPurchase) {
      throw new BadRequestException(`Minimum purchase required is ₹${coupon.minPurchase}`);
    }

    let discountAmount = 0;

    if (coupon.discountType === 'PERCENTAGE') {
      discountAmount = (cart.totalPrice * coupon.discountValue) / 100;
      if (coupon.maxDiscount) {
        discountAmount = Math.min(discountAmount, coupon.maxDiscount);
      }
    } else {
      discountAmount = coupon.discountValue;
    }

    const newTotal = cart.totalPrice - discountAmount;
    cart.totalPrice = parseFloat(newTotal.toFixed(2));
    cart.coupon = coupon;

    userCoupon.status = CouponStatus.USED;
    userCoupon.usedAt = new Date();
    await this.userCouponRepo.save(userCoupon);
    await this.cartRepo.save(cart);

    return {
      message: 'Coupon applied successfully',
      discountAmount,
      newTotal: cart.totalPrice,
    };
  }


 async getCartItems(accountId: string) {
    const cart = await this.cartRepo
      .createQueryBuilder('cart')
      .leftJoinAndSelect('cart.cartItems', 'cartItems')
      .leftJoinAndSelect('cartItems.product', 'product')
      .where('cart.accountId = :accountId', { accountId })
      .select([
        'cart.id',
        'cart.accountId',
        'cart.totalPrice',
        'cart.createdAt',
        'cartItems.id',
        'cartItems.quantity',
        'cartItems.price',
        'cartItems.subTotalPrice',
        'cartItems.createdAt',
        'product.id',
        'product.name',
        'product.description',
        'product.imageUrl',
        'product.quantity',
      ])
      .getOne();

    if (!cart) {
      throw new NotFoundException('Cart not found');
}
    const subTotal = cart.cartItems.reduce((sum, item) => sum + Number(item.subTotalPrice), 0);
    const gst = +(subTotal * 0.18).toFixed(2);
    const deliveryCharge = 0;
    const totalAmount = +(subTotal + gst + deliveryCharge).toFixed(2);

    return {
     // cartId: cart.id,
  ...cart,
      subTotal,
      gst,
      deliveryCharge,
      totalAmount,
    };

  }

 async updateCartItemQuantity(
  cartItemId: string,
  dto: UpdateCartItemQtyDto,
  accountId: string,
) {
  const item = await this.cartItemrepo.findOne({
    where: { id: cartItemId },
    relations: ['cart', 'product'],
  });

  if (!item) throw new NotFoundException('Cart item not found');

  if (
    item.accountId !== accountId ||
    item.cart.accountId !== accountId
  ) {
    throw new ForbiddenException('You are not allowed to update this item');
  }

  
  if (dto.quantity <= 0) {
    await this.cartItemrepo.remove(item);
    const summary = await this.recalculateCartTotal(item.cart.id);
    return { message: 'Item removed from cart', summary };
  }

 
  item.quantity = dto.quantity;
  item.subTotalPrice = +(item.price * item.quantity).toFixed(2);
  await this.cartItemrepo.save(item);

  
  const summary = await this.recalculateCartTotal(item.cart.id);

  return {
    message: 'Quantity updated',
    item: {
      id: item.id,
      quantity: item.quantity,
      price: item.price,
      subTotal: item.subTotalPrice,
      productName: item.product?.name 
    },
    summary,
  };
}

  async removes(id: string, accountId: string) {
    const cartItem = await this.cartItemrepo.findOne({
      where: { id: id, accountId: accountId },
      relations: ['cart']
    });

    if (!cartItem) {
      throw new NotFoundException('Cart item not found');
    }

    const cartId = cartItem.cart.id;
    await this.cartItemrepo.remove(cartItem);
    const summary = await this.recalculateCartTotal(cartId);

    return {
      message: 'Cart item removed successfully',
      summary
    };
  }
  


  async recalculateCartTotal(cartId: string) {
  const cartItems = await this.cartItemrepo.find({
    where: { cart: { id: cartId } }, 
  });

  const subTotal = cartItems.reduce(
    (sum, item) => sum + Number(item.subTotalPrice),
    0,
  );

  const gst = +(subTotal * 0.18).toFixed(2);
  const deliveryCharge = 0;
  const total = +(subTotal + gst + deliveryCharge).toFixed(2);

  
  await this.cartRepo.update(cartId, { totalPrice: subTotal });

  return { subTotal, gst, deliveryCharge, total };
}



}
