import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { CategoryDto, PaginationSDto, StatusDto, StatusPaginationDto } from './dto/category.dto';
import { Category } from './entities/category.entity';
import { DefaultStatus, UserRole } from 'src/enum';
import { Product } from '../product/entities/product.entity';
import { Brand } from '../brand/entities/brand.entity';
import { Wishlist } from '../wishlist/entities/wishlist.entity';


@Injectable()
export class CategoryService {
  constructor(
    @InjectRepository(Category) private readonly categoryRepository: Repository<Category>,
  ) { }

  async create(dto: CategoryDto) {
    const category = await this.categoryRepository.findOne({ where: { name: dto.name } });
    if (category) {
      throw new ConflictException('Category already exists!');
    }
    const obj = Object.create(dto);
    return this.categoryRepository.save(obj);
  }

  async findAll(dto: StatusPaginationDto) {
    const keyword = dto.keyword || '';
    const status = dto.status;

    const query = this.categoryRepository
      .createQueryBuilder('category')
      .select([
        'category.id',
        'category.name',
        'category.image',
        'category.imageName',
        'category.status',
        'category.createdAt',
      ])
      .andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :name', { name: `%${keyword}%` })
        }),
      )

    if (dto.status) {
      query.andWhere('category.status = :status', { status: dto.status });
    }
    const [result, total] = await query
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findByUser(dto: PaginationSDto) {
    const keyword = dto.keyword || '';
    const query = this.categoryRepository.createQueryBuilder('category')
      .select([
        'category.id',
        'category.name',
        'category.image',
        'category.imageName',
        'category.status',
        'category.createdAt',
      ])
      .innerJoin('category.products', 'product')
      .where('category.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :name', { name: `%${keyword}%` })
        }),
      )
    if (dto.brandId) {
      query.andWhere('product.brandId = :brandId', { brandId: dto.brandId });
    }

    const [result, total] = await query
      .distinct(true)
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findOne(id: string) {
    const category = await this.categoryRepository.findOne({ where: { id: id } });
    if (!category) {
      throw new NotFoundException('Category not found!');
    }
    return category;
  }
  async getCrazyDealCategories(dto: PaginationSDto) {
    const query = this.categoryRepository
      .createQueryBuilder('category')
      .leftJoinAndSelect('category.products', 'product')
      .select([
        'category.id',
        'category.name',
        'category.crzyImage',
        'category.crzyImgName'
      ])
      .where('category.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere('product.isCrazyDeal = true');

    const [result, total] = await query
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async image(image: string, result: Category) {
    const obj = Object.assign(result, {
      image: process.env.RN_CDN_LINK + image,
      imageName: image,
    });
    return await this.categoryRepository.save(obj);
  }

  async saveCezyImage(image: string, result: Category) {
    const obj = Object.assign(result, {
      crzyImage: process.env.RN_CDN_LINK + image,
      crzyImgName: image,
    });
    return await this.categoryRepository.save(obj);
  }

  async status(id: string, dto: StatusDto) {
    const menu = await this.categoryRepository.findOne({ where: { id: id } });
    if (!menu) {
      throw new NotFoundException('Category not found!');
    }
    const obj = Object.assign(menu, dto);
    return this.categoryRepository.save(obj);
  }

  async update(id: string, dto: CategoryDto) {
    try {
      const category = await this.categoryRepository.findOne({ where: { id } });
      if (!category) {
        throw new NotFoundException('Category not found!');
      }
      const obj = Object.assign(category, dto);
      return this.categoryRepository.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either category exists or invalid name!',
      );
    }
  }

async remove(id: string) {
  const category = await this.categoryRepository.findOne({ where: { id } });

  if (!category) {
    throw new NotFoundException('Category not found');
  }
  const product = await this.categoryRepository
    .createQueryBuilder('category')
    .leftJoinAndSelect('category.products', 'product')
    .where('category.id = :id', { id })
    .getOne();
  if (product.products.length > 0) {
    throw new NotAcceptableException('Category has products');
  }
  await this.categoryRepository.delete(category.id);
  return { message: 'Category deleted successfully' };
}


}
