import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CreateCouponDto } from './dto/create-coupon.dto';
import { UpdateCouponDto } from './dto/update-coupon.dto';
import { Coupon } from './entities/coupon.entity';
import { UserCoupon } from './entities/user-coupon.entity';
import { CouponStatus, UserRole } from '../enum';
import { NotificationsService } from '../notifications/notifications.service';
import { Account } from '../account/entities/account.entity';

@Injectable()
export class CouponService {
  constructor(
    @InjectRepository(Coupon)
    private readonly couponRepo: Repository<Coupon>,
    @InjectRepository(UserCoupon)
    private readonly userCouponRepo: Repository<UserCoupon>,
    @InjectRepository(Account)
    private readonly accountRepo: Repository<Account>,
    private readonly notificationService: NotificationsService,
  ) {}

  async assignCouponToUsers(couponId: string, userIds: string[], sendNotification: boolean = true) {
    const coupon = await this.couponRepo.findOne({ where: { id: couponId } });
    if (!coupon) {
      throw new NotFoundException('Coupon not found');
    }

    const userCoupons = userIds.map(userId =>
      this.userCouponRepo.create({
        accountId: userId,
        couponId: couponId,
        status: CouponStatus.ACTIVE
      })
    );

    await this.userCouponRepo.save(userCoupons);

    if (sendNotification) {
      for (const userId of userIds) {
        await this.notificationService.create({
          title: 'New Coupon Available!',
          desc: `You have received a new coupon: ${coupon.code}. ${coupon.description || 'Use it now and save money!'}`,
          accountId: userId
        });
      }
    }

    return {
      message: `Coupon assigned to ${userIds.length} user(s) successfully`,
      assignedUsers: userIds.length,
      couponCode: coupon.code
    };
  }

  async assignCouponToAllUsers(couponId: string, userRole?: UserRole, sendNotification: boolean = true) {
    const coupon = await this.couponRepo.findOne({ where: { id: couponId } });
    if (!coupon) {
      throw new NotFoundException('Coupon not found');
    }

    const whereCondition = userRole ? { roles: userRole } : {};
    const users = await this.accountRepo.find({ where: whereCondition, select: ['id'] });
    const userIds = users.map(user => user.id);

    if (userIds.length === 0) {
      return { message: 'No users found to assign coupon' };
    }

    return this.assignCouponToUsers(couponId, userIds, sendNotification);
  }

  async getUserCoupons(accountId: string) {
    return await this.userCouponRepo.find({
      where: { accountId, status: CouponStatus.ACTIVE },
      relations: ['coupon'],
    });
  }

  async create(createCouponDto: CreateCouponDto) {
    const coupon = Object.create({
      ...createCouponDto,
      expiresAt: createCouponDto.expiresAt ? new Date(createCouponDto.expiresAt) : null
    });
    return await this.couponRepo.save(coupon);
  }

  async findAll() {
    return await this.couponRepo.find({
      order: { createdAt: 'DESC' }
    });
  }

  async findOne(id: string) {
    const coupon = await this.couponRepo.findOne({ where: { id } });
    if (!coupon) {
      throw new NotFoundException('Coupon not found');
    }
    return coupon;
  }

  async update(id: string, updateCouponDto: UpdateCouponDto) {
    const coupon = await this.findOne(id);
    Object.assign(coupon, {
      ...updateCouponDto,
      expiresAt: updateCouponDto.expiresAt ? new Date(updateCouponDto.expiresAt) : coupon.expiresAt
    });
    return await this.couponRepo.save(coupon);
  }

  async remove(id: string) {
    const coupon = await this.findOne(id);
    await this.couponRepo.remove(coupon);
    return { message: 'Coupon deleted successfully' };
  }
}
