import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards } from '@nestjs/common';
import { CreateDeliveryDetailDto } from './dto/create-delivery-detail.dto';
import { UpdateDeliveryDetailDto } from './dto/update-delivery-detail.dto';
import { AuthGuard } from '@nestjs/passport';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { DeliveryAddressService } from './delivery-address.service';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';

@Controller('delivery-details')
export class DeliveryAddressController {
  constructor(private readonly deliveryAddressService: DeliveryAddressService) { }

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  create(
    @Body() dto: CreateDeliveryDetailDto,
    @CurrentUser() user: Account,
  ) {
    return this.deliveryAddressService.create(dto, user.id);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER,)
  async getWishlistItems(@CurrentUser() user: Account) {
    const accountId = user.id;
    return this.deliveryAddressService.getDeliveryDetails(accountId);
  }

   @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'delivery_address'])
  async getDeliveryDetails(@CurrentUser() user: Account) {
    const accountId = user.id;
    return this.deliveryAddressService.getDeliveryDetails(accountId);
  }

  @Get('default')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  async getDefault(@CurrentUser() user: Account) {
    return this.deliveryAddressService.getDefaultAddress(user.id);
  }


  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  update(
    @Param('id') id: string,
    @Body() dto: UpdateDeliveryDetailDto,
    @CurrentUser() user: Account,
  ) {
    return this.deliveryAddressService.update(id, dto, user.id);
  }

  @Patch('default/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  setDefault(@Param('id') id: string, @CurrentUser() user: Account) {
    return this.deliveryAddressService.setDefault(id, user.id);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  remove(@Param('id') id: string, @CurrentUser() user: Account) {
    const accountId = user.id;
    return this.deliveryAddressService.removes(id, accountId);
  }
}
