import { Injectable, NotFoundException } from '@nestjs/common';
import { CreateDeliveryDetailDto } from './dto/create-delivery-detail.dto';
import { UpdateDeliveryDetailDto } from './dto/update-delivery-detail.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { deliveryAddress } from './entities/delivery-address.entity';


@Injectable()
export class DeliveryAddressService {
  constructor(
    @InjectRepository(deliveryAddress)
    private readonly deliveryRepo: Repository<deliveryAddress>,
  ) { }

async create(dto: CreateDeliveryDetailDto, accountId: string) {
  
  const addressCount = await this.deliveryRepo.count({ where: { accountId } });

 
  const newAddress = this.deliveryRepo.create({
    ...dto,
    accountId,
    isDefault: addressCount === 0, 
  });


  return this.deliveryRepo.save(newAddress);
}
  async getDeliveryDetails(accountId: string) {
    const [result, total] = await this.deliveryRepo
      .createQueryBuilder('delivery')
      .where('delivery.accountId = :accountId', { accountId })
      .select([
        'delivery.id',
        'delivery.accountId',
        'delivery.name',
        'delivery.address',
        'delivery.landMark',
        'delivery.city',
        'delivery.state',
        'delivery.pin',
        'delivery.phNumber',
        'delivery.isDefault',
      ])
      .getManyAndCount();

    if (!result || result.length === 0) {
      throw new NotFoundException('Delivery details not found');
    }

    return {
      result,
      total,
    };
  }
  async getDeliveryDetail(accountId: string) {
    const [result, total] = await this.deliveryRepo
      .createQueryBuilder('delivery')
      .where('delivery.accountId = :accountId', { accountId })
      .select([
        'delivery.id',
        'delivery.accountId',
        'delivery.name',
        'delivery.address',
        'delivery.landMark',
        'delivery.city',
        'delivery.state',
        'delivery.pin',
        'delivery.phNumber',
        'delivery.isDefault',
      ])
      .getManyAndCount();

    if (!result || result.length === 0) {
      throw new NotFoundException('Delivery details not found');
    }

    return {
      result,
      total,
    };
  }



  async getDefaultAddress(accountId: string) {
    const defaultAddress = await this.deliveryRepo.findOne({
      where: { accountId, isDefault: true },
      select: [
        'id',
        'accountId',
        'name',
        'address',
        'landMark',
        'city',
        'state',
        'pin',
        'phNumber',
        'isDefault',
      ],
    });

    if (!defaultAddress) {
      throw new NotFoundException('No default address found');
    }

    return defaultAddress;
  }


  async update(id: string, dto: UpdateDeliveryDetailDto, accountId: string) {
    const address = await this.deliveryRepo.findOne({
      where: { id, accountId },
    });

    if (!address) {
      throw new NotFoundException('Delivery address not found or unauthorized');
    }

    await this.deliveryRepo.update(id, dto);

    return { message: 'Delivery address updated successfully' };
  }

  async setDefault(id: string, accountId: string) {
    const address = await this.deliveryRepo.findOneBy({ id, accountId });
    if (!address) {
      throw new NotFoundException('Address not found or unauthorized');
    }

    await this.deliveryRepo.update({ accountId }, { isDefault: false });
    await this.deliveryRepo.update(id, { isDefault: true });

    return { message: 'Default address set successfully' };
  }


  async removes(id: string, accountId: string) {
    const detail = await this.deliveryRepo.findOne({ where: { id: id, accountId: accountId } });
    if (!detail) {
      throw new NotFoundException('Delivery address not found');
    }
    if(detail.isDefault === true){
      throw new NotFoundException('Default address cannot be removed');
    }
    
    // Since we're now storing the full address in the order table,
    // we can safely delete the delivery address even if it's used in orders
    await this.deliveryRepo.remove(detail);
    return ({ message: 'Delivery address removed successfully' });
  }


}
