import { Injectable, Logger } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Setting } from 'src/settings/entities/setting.entity';
import { DefaultStatus, OrderStatus, UserRole } from 'src/enum';
import { Account } from 'src/account/entities/account.entity';
import { MailerService } from '@nestjs-modules/mailer';
import { OrderItem } from 'src/order-item/entities/order-item.entity';

@Injectable()
export class NodeMailerService {
  private readonly logger = new Logger(NodeMailerService.name);

  constructor(
    @InjectRepository(Setting)
    private readonly settingRepo: Repository<Setting>,
    @InjectRepository(Account)
    private readonly accountRepo: Repository<Account>,
    private readonly mailerService: MailerService
  ) { }

  private async getAdminEmail(): Promise<string> {
    try {
      const settings = await this.settingRepo.findOne({
        where: { status: DefaultStatus.ACTIVE },
      });
      return settings?.email || process.env.RN_SMTP_USER;
    } catch (error) {
      this.logger.error(`Error getting admin email: ${error.message}`);
      return process.env.SMTP_USER;
    }
  }



  async sendOrderStatusUpdate(userEmail: string, userName: string, orderId: string, status: string, estimatedDeliveryDate: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = `Order #${orderId.slice(0, 8)} Status Update`;

    let statusMessage = '';
    switch (status) {
      case OrderStatus.SHIPPED:
        statusMessage = 'Your order has been shipped and is on its way to you!';
        break;
      case OrderStatus.DELIVERED:
        statusMessage = 'Your order has been delivered. We hope you enjoy your purchase!';
        break;
      case OrderStatus.CANCELED:
        statusMessage = 'Your order has been canceled as requested.';
        break;
      case OrderStatus.PENDING_PAYMENT:
        statusMessage = 'Your order is pending payment. Please complete the payment to process your order.';
        break;
      case OrderStatus.PLACED:
        statusMessage = 'Your order has been placed successfully and is being processed.';
        break;
      case OrderStatus.COMPLETED:
        statusMessage = 'Your order has been completed. Thank you for shopping with us!';
        break;
      default:
        statusMessage = `Your order status has been updated to ${status}.`;
    }

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">Order Status Update</h2>
        <p>Hello ${userName},</p>
        <p>${statusMessage}</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <h3 style="margin-top: 0;">Order #${orderId.slice(0, 8)}</h3>
          <p><strong>Status:</strong> ${status}</p>
          <p><strong>Estimated Delivery:</strong> ${estimatedDeliveryDate}</p>
        </div>
        
        <p>You can track your order status in your account dashboard.</p>
        <p style="color: #777; font-size: 0.9em;">Thank you for shopping with us!</p>
      </div>
    `;

    return this.sendEmail(userEmail, subject, html, adminEmail);
  }

  async sendOrderConfirmation(userEmail: string, order: any, userName: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = `Order Confirmation #${order.id.slice(0, 8)}`;

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">Your Order is Confirmed!</h2>
        <p>Thank you for your order. We're processing it now and will notify you when it ships.</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <h3 style="margin-top: 0;">Order #${order.id.slice(0, 8)}</h3>
          <p><strong>Estimated Delivery:</strong> ${order.estimatedDeliveryDate}</p>
          <p><strong>Delivery Address:</strong><br>${order.fullDeliveryAddress}</p>
          <p><strong>Payment Method:</strong> ${order.paymentMethod}</p>
        </div>
        
        <h3>Order Summary</h3>
        <p><strong>Subtotal:</strong> ₹${order.subTotal}</p>
        <p><strong>GST:</strong> ₹${order.gstAmount}</p>
        <p><strong>Delivery Charge:</strong> ₹${order.deliveryCharge}</p>
        <p><strong>Total Amount:</strong> ₹${order.totalAmount}</p>
        
        <p style="margin-top: 20px;">If you have any questions about your order, please contact our customer service.</p>
        <p style="color: #777; font-size: 0.9em;">Thank you for shopping with us!</p>
      </div>
    `;

    return this.sendEmail(userEmail, subject, html, adminEmail);
  }

  async sendNewOrderNotificationToAdmin(order: any, customerName: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = `New Order #${order.id.slice(0, 8)} Received`;

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">New Order Received</h2>
        <p>A new order has been placed by ${customerName}.</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <h3 style="margin-top: 0;">Order #${order.id.slice(0, 8)}</h3>
          <p><strong>Customer:</strong> ${customerName}</p>
          <p><strong>Payment Method:</strong> ${order.paymentMethod}</p>
          <p><strong>Total Amount:</strong> ₹${order.totalAmount}</p>
          <p><strong>Delivery Address:</strong><br>${order.fullDeliveryAddress}</p>
        </div>
        
        <p style="margin-top: 20px;">Please process this order at your earliest convenience.</p>
      </div>
    `;

    return this.sendEmail(adminEmail, subject, html, adminEmail);
  }

  async sendPriceDropNotification(userEmail: string, userName: string, productName: string, oldPrice: number, newPrice: number, productId: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = 'Price Drop Alert!';

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">Price Drop Alert!</h2>
        <p>Hello ${userName},</p>
        <p>Good news! A product in your wishlist has dropped in price:</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <h3 style="margin-top: 0;">${productName}</h3>
          <p><strong>New Price:</strong> ₹${newPrice}</p>
          <p><strong>Old Price:</strong> ₹${oldPrice}</p>
          <p><strong>You Save:</strong> ₹${oldPrice - newPrice}</p>
        </div>
        
        <p>Grab it before the price changes again!</p>
        <p style="color: #777; font-size: 0.9em;">Thank you for shopping with us!</p>
      </div>
    `;

    return this.sendEmail(userEmail, subject, html, adminEmail);
  }
  async sendCartReminderEmail(userEmail: string, userName: string, cartItems: any[]) {
    const adminEmail = await this.getAdminEmail();
    const subject = 'Your Cart is Waiting for You';

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">Your Cart is Waiting!</h2>
        <p>Hello ${userName},</p>
        <p>We noticed you have items in your cart that are waiting to be purchased.</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <h3 style="margin-top: 0;">Items in Your Cart:</h3>
          <ul>
            ${cartItems.map(item => `
              <li>
                <strong>${item.product?.name || 'Product'}</strong> - Quantity: ${item.quantity} - ₹${item.price}
              </li>
            `).join('')}
          </ul>
        </div>
        
        <p>Complete your purchase now before these items go out of stock!</p>
        <p style="color: #777; font-size: 0.9em;">Thank you for shopping with us!</p>
      </div>
    `;

    return this.sendEmail(userEmail, subject, html, adminEmail);
  }

  async sendCustomNotification(userEmail: string, userName: string, title: string, message: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = title;

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">${title}</h2>
        <p>Hello ${userName},</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <p>${message}</p>
        </div>
        
        <p style="color: #777; font-size: 0.9em;">Thank you for being a valued customer!</p>
      </div>
    `;

    return this.sendEmail(userEmail, subject, html, adminEmail);
  }
  async sendCustomNotificationAdmin(title: string, message: string) {
    const adminEmail = await this.getAdminEmail();
    const subject = title;

    const html = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee;">
        <h2 style="color: #333;">${title}</h2>
        <p>Hello Admin,</p>
        
        <div style="background-color: #f9f9f9; padding: 15px; margin: 15px 0; border-radius: 5px;">
          <p>${message}</p>
        </div>
        
        <p style="color: #777; font-size: 0.9em;">Please take necessary action.</p>
      </div>
    `;

    return this.sendEmail(adminEmail, subject, html, adminEmail);
  }

  private async sendEmail(to: string, subject: string, html: string, from: string) {
    try {
      if (!to) {
        this.logger.warn('No recipient email provided');
        return { success: false, error: 'No recipient email provided' };
      }

      const info = await this.mailerService.sendMail({
        to,
        subject,
        html,
      });

      this.logger.log(`Email sent: ${info.messageId}`);
      return { success: true, messageId: info.messageId };
    } catch (error) {
      this.logger.error(`Error sending email: ${error.message}`);
      return { success: false, error: error.message };
    }
  }
}