import { HttpService } from '@nestjs/axios';
import { BadRequestException, Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, LessThanOrEqual } from 'typeorm';
import { Cron, CronExpression } from '@nestjs/schedule';
import { Notification } from './entities/notification.entity';
import { CartItem } from '../cart-item/entities/cart-item.entity';
import { Product } from '../product/entities/product.entity';
import { NotificationType, ProductStatus, UserRole } from '../enum';
import { Wishlist } from '../wishlist/entities/wishlist.entity';
import { Account } from '../account/entities/account.entity';
import { PaginationDto } from './dto/notification.dto';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';

@Injectable()
export class NotificationsService {
  constructor(
    @InjectRepository(Notification)
    private readonly repo: Repository<Notification>,
    @InjectRepository(CartItem)
    private readonly cartItemRepo: Repository<CartItem>,
    @InjectRepository(Product)
    private readonly productRepo: Repository<Product>,
    @InjectRepository(Wishlist)
    private readonly wishlistRepo: Repository<Wishlist>,
    @InjectRepository(Account)
    private readonly accountRepo: Repository<Account>,
    private readonly httpService: HttpService,
    private readonly nodeMailerService: NodeMailerService
  ) { }

  async create(notificationData: {
    title: string;
    desc: string;
    accountId?: string;
    type?: NotificationType;
  }) {
    const notification = this.repo.create({
      title: notificationData.title,
      desc: notificationData.desc,
      accountId: notificationData.accountId,
      type: notificationData.type || NotificationType.SYSTEM,
      read: false
    });

    return this.repo.save(notification);
  }

  async sendCartAbandonmentNotifications() {
    const sevenDaysAgo = new Date();
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

    const abandonedCarts = await this.cartItemRepo
      .createQueryBuilder('cartItem')
      .select('DISTINCT cartItem.accountId', 'accountId')
      .where('cartItem.createdAt <= :sevenDaysAgo', { sevenDaysAgo })
      .getRawMany();

    for (const cart of abandonedCarts) {
      await this.create({
        title: 'Amazing Deals Waiting!',
        desc: 'You left some amazing deals in your cart. Complete your purchase now before they\'re gone!',
        accountId: cart.accountId,
        type: NotificationType.CART_ABANDONMENT
      });
    }

    return { message: `Sent ${abandonedCarts.length} cart abandonment notifications` };
  }

  async sendCartReminderNotifications(sendEmail: boolean = true) {
    const oneDayAgo = new Date();
    oneDayAgo.setDate(oneDayAgo.getDate() - 1);

    const carts = await this.cartItemRepo
      .createQueryBuilder('cartItem')
      .leftJoinAndSelect('cartItem.cart', 'cart')
      .leftJoinAndSelect('cartItem.product', 'product')
      .leftJoinAndSelect('cart.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .where('cartItem.createdAt <= :oneDayAgo', { oneDayAgo })
      .andWhere('cart.totalPrice > 0')
      .getMany();

    const accountCarts = {};

    carts.forEach(item => {
      if (!accountCarts[item.cart.accountId]) {
        accountCarts[item.cart.accountId] = {
          items: [],
          account: item.cart.account
        };
      }
      accountCarts[item.cart.accountId].items.push(item);
    });

    for (const accountId in accountCarts) {
      const cartData = accountCarts[accountId];
      const account = cartData.account;
      const userEmail = account?.userDetail?.[0]?.email;
      const userName = account?.userDetail?.[0]?.name || 'Valued Customer';

      await this.create({
        title: 'Your Cart is Waiting!',
        desc: `You have ${cartData.items.length} item(s) in your cart waiting to be purchased.`,
        accountId,
        type: NotificationType.CART_REMINDER
      });

      if (sendEmail && userEmail) {
        try {
          await this.nodeMailerService.sendCartReminderEmail(
            userEmail,
            userName,
            cartData.items
          );
        } catch (error) {
          console.error('Failed to send cart reminder email:', error);
        }
      }
    }

    return { message: `Sent ${Object.keys(accountCarts).length} cart reminder notifications` };
  }

  async sendPriceDropNotifications(productId: string, newPrice: number, oldPrice: number, sendEmail: boolean = true) {
    const product = await this.productRepo.findOne({ where: { id: productId } });
    if (!product) return;

    const wishlists = await this.wishlistRepo
      .createQueryBuilder('wishlist')
      .leftJoinAndSelect('wishlist.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .where('wishlist.productId = :productId', { productId })
      .getMany();

    const notifications = wishlists.map(wishlist => ({
      title: 'Price Drop Alert!',
      desc: `${product.name} is now ₹${newPrice} (was ₹${oldPrice}). Grab it before the price changes!`,
      accountId: wishlist.accountId,
      read: false,
      createdAt: new Date(),
      type: NotificationType.PRICE_DROP
    }));

    await this.repo.save(notifications);

    if (sendEmail) {
      for (const wishlist of wishlists) {
        const userEmail = wishlist.account?.userDetail?.[0]?.email;
        if (userEmail) {
          try {
            await this.nodeMailerService.sendPriceDropNotification(
              userEmail,
              wishlist.account?.userDetail?.[0]?.name || 'Valued Customer',
              product.name,
              oldPrice,
              newPrice,
              product.id
            );
          } catch (error) {
            console.error('Failed to send price drop email:', error);
          }
        }
      }
    }

    return { message: `Sent ${notifications.length} price drop notifications` };
  }

  async sendBackInStockNotifications(productId: string) {
    const product = await this.productRepo.findOne({ where: { id: productId } });
    if (!product || product.status !== ProductStatus.IN_STOCK) return;

    const wishlists = await this.wishlistRepo.find({ where: { productId } });

    const notifications = wishlists.map(wishlist => ({
      title: 'Back in Stock!',
      desc: `${product.name} is back in stock. Get it before it's gone again!`,
      accountId: wishlist.accountId,
      read: false,
      createdAt: new Date(),
      type: NotificationType.BACK_IN_STOCK
    }));

    await this.repo.save(notifications);
    return { message: `Sent ${notifications.length} back-in-stock notifications` };
  }

  async sendNewProductNotifications(productId: string) {
    const product = await this.productRepo.findOne({ where: { id: productId } });
    if (!product) return;

    const wishlists = await this.wishlistRepo
      .createQueryBuilder('wishlist')
      .innerJoin('wishlist.product', 'product')
      .where('product.categoryId = :categoryId', { categoryId: product.categoryId })
      .select('DISTINCT wishlist.accountId', 'accountId')
      .getRawMany();

    const notifications = wishlists.map(w => ({
      title: 'New Product Alert!',
      desc: `Check out our new product: ${product.name}. Be the first to get it!`,
      accountId: w.accountId,
      read: false,
      createdAt: new Date(),
      type: NotificationType.NEW_PRODUCT
    }));

    await this.repo.save(notifications);
    return { message: `Sent ${notifications.length} new product notifications` };
  }

  async sendLowStockNotification(productId: string) {
    const product = await this.productRepo.findOne({ where: { id: productId } });
    if (!product) return;

    const admins = await this.accountRepo.find({ where: { roles: UserRole.ADMIN } });
    for (const admin of admins) {
      await this.create({
        title: 'Low Stock Alert!',
        desc: `${product.name} is running low on stock. Current stock: ${product.quantity}. Please restock soon.`,
        accountId: admin.id,
        type: NotificationType.SYSTEM
      });
    }
  }

  @Cron(CronExpression.EVERY_HOUR)
  async checkLowStockProducts() {
    const threshold = 5;
    const products = await this.productRepo.find({
      where: {
        quantity: LessThanOrEqual(threshold),
        status: ProductStatus.IN_STOCK
      }
    });

    for (const product of products) {
      await this.sendLowStockNotification(product.id);
    }

    return { message: `Checked ${products.length} low stock products` };
  }

  @Cron(CronExpression.EVERY_DAY_AT_10AM)
  async scheduledCartAbandonmentNotifications() {
    return await this.sendCartAbandonmentNotifications();
  }

  @Cron(CronExpression.EVERY_DAY_AT_3PM)
  async scheduledCartReminderNotifications() {
    return await this.sendCartReminderNotifications(true);
  }


  async findAll(dto: PaginationDto, accountId: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('notification')
      .where('notification.accountId = :accountId OR notification.accountId IS NULL', { accountId })
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy('notification.createdAt', 'DESC')
      .getManyAndCount();
    return { result, total };
  }

  async update(id: number, accountId: string, status: boolean) {
    const notifs = await this.repo.findOne({ where: { id, accountId } });
    if (!notifs) throw new NotFoundException('Notification not found!');
    notifs.read = status;
    return this.repo.save(notifs);
  }

  async markAllAsRead(accountId: string) {
    if (!accountId) {
      throw new BadRequestException('accountId is required');
    }

    try {
      const result = await this.repo
        .createQueryBuilder()
        .update()
        .set({ read: true })
        .where('accountId = :accountId', { accountId })
        .andWhere('read = :read', { read: false })
        .execute();

      return {
        message: 'All notifications marked as read',
        count: result.affected || 0
      };
    } catch (error) {
      console.error('Error marking notifications as read:', error);
      return { message: 'All notifications marked as read', count: 0 };
    }
  }


  async clearAllNotifications(accountId: string) {
    try {
      const result = await this.repo.delete({ accountId });
      return { message: 'All notifications cleared', count: result.affected || 0 };
    } catch (error) {
      return { message: 'All notifications cleared', count: 0 };
    }
  }

  async getUnreadCount(accountId: string) {
    const count = await this.repo.count({ where: { accountId, read: false } });
    return { count };
  }

  async sendManualNotification(accountIds: string[], title: string, message: string, sendEmail: boolean = false) {
    const notifications = [];

    for (const accountId of accountIds) {
      notifications.push({
        title,
        desc: message,
        accountId,
        read: false,
        createdAt: new Date(),
        type: NotificationType.MANUAL
      });
    }

    await this.repo.save(notifications);

    if (sendEmail) {
      const accounts = await this.accountRepo
        .createQueryBuilder('account')
        .leftJoinAndSelect('account.userDetail', 'userDetail')
        .where('account.id IN (:...accountIds)', { accountIds })
        .getMany();

      for (const account of accounts) {
        const userEmail = account.userDetail?.[0]?.email;
        const userName = account.userDetail?.[0]?.name || 'Valued Customer';
        if (userEmail) {
          try {
            await this.nodeMailerService.sendCustomNotification(
              userEmail,
              userName,
              title,
              message
            );
          } catch (error) {
            console.error('Failed to send manual notification email:', error);
          }
        }
      }
    }

    return { message: `Sent ${notifications.length} manual notifications` };
  }

  async sendCustomNotification(accountId: string, title: string, message: string, sendEmail: boolean = false) {
    const notification = await this.create({
      title,
      desc: message,
      accountId,
      type: NotificationType.MANUAL
    });

    if (sendEmail) {
      const account = await this.accountRepo
        .createQueryBuilder('account')
        .leftJoinAndSelect('account.userDetail', 'userDetail')
        .where('account.id = :accountId', { accountId })
        .getOne();

      if (account) {
        const userEmail = account.userDetail?.[0]?.email;
        const userName = account.userDetail?.[0]?.name || 'Valued Customer';

        if (userEmail) {
          try {
            await this.nodeMailerService.sendCustomNotification(
              userEmail,
              userName,
              title,
              message
            );
          } catch (error) {
            console.error('Failed to send custom notification email:', error);
          }
        }
      }
    }

    return { message: 'Custom notification sent successfully', notification };
  }

  async sendBulkNotification(userRole: UserRole, title: string, message: string, sendEmail: boolean = false) {
    const accounts = await this.accountRepo.find({ where: { roles: userRole } });
    const accountIds = accounts.map(account => account.id);

    return this.sendManualNotification(accountIds, title, message, sendEmail);
  }
}