import {
  Controller, Get, Post, Body, Patch, Param, Query,
  UseGuards, BadRequestException,
  Res,
  NotFoundException
} from '@nestjs/common';
import { Response } from 'express';
import { OrderService } from './order.service';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { UserRole, PermissionAction } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import {
  CreateOrderDto,
  DirectBuyDto,
  PaginationDto,
  StatusDto,
  VerifyPaymentDto
} from './dto/order.dto';
import { AdminCreateOrderDto } from './dto/admin-create-order.dto';
import { ReturnOrderDto } from './dto/return-order.dto';
import { ProcessRefundDto } from './dto/process-refund.dto';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';

@Controller('order')
export class OrderController {
  constructor(private readonly orderService: OrderService) { }

  @Post('verify-payment')
  @UseGuards(AuthGuard('jwt'))
  async verifyPayment(
    @CurrentUser() user: Account,
    @Body() dto: VerifyPaymentDto
  ) {
    return this.orderService.verifyPaymentSignature(dto, user.id);
  }
  @Post('admin/place-order')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'order'])
  async adminPlaceOrder(
    @Body() dto: AdminCreateOrderDto,
    @CurrentUser() admin: Account
  ) {
    return this.orderService.adminPlaceOrder(dto, admin.id);
  }

  @Post('checkout-cart')
  @UseGuards(AuthGuard('jwt'))
  async checkoutCart(
    @CurrentUser() user: Account,
    @Body() dto: CreateOrderDto
  ) {
    return this.orderService.handleCartCheckout(user.id, dto.paymentMethod, dto);
  }

  @Get('preview-user')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async previewUserBuyNow(
    @Query('productId') productId: string,
    @Query('quantity') quantity: number,
    @CurrentUser() user: Account
  ) {
    return this.orderService.previewBuyNowUser(
      { productId, quantity: +quantity },
      user.id
    );
  }

  @Get('preview-retailer')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async previewRetailerBuyNow(
    @Query('productId') productId: string,
    @Query('quantity') quantity: number,
    @CurrentUser() user: Account
  ) {
    return this.orderService.previewBuyNowRetailer(
      { productId, quantity: +quantity },
      user.id
    );
  }

  @Post('user/place-direct-order')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async placeDirectOrderUser(
    @Body() dto: DirectBuyDto,
    @CurrentUser() user: Account
  ) {
    return this.orderService.userHandleDirectOrder(dto, user.id);
  }

  @Post('retailer/place-direct-order')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async placeDirectOrderRetailer(
    @Body() dto: DirectBuyDto,
    @CurrentUser() user: Account
  ) {
    return this.orderService.retailerHandleDirectOrder(dto, user.id);
  }

  @Get('admin/all-orders')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'order'])
  findAll(@Query() dto: PaginationDto) {
    return this.orderService.findAll(dto);
  }

  @Get('admin/order/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'order'])
  async getOrderByyAdmin(
    @Param('id') id: string,
    @Query() dto: PaginationDto
  ) {
    return this.orderService.getOrdersOfUserByAdmin(id, dto);
  }

  @Get('user')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  async getOrderItems(
    @CurrentUser() user: Account,
    @Query() dto: PaginationDto
  ) {
    return this.orderService.getOrders(user.id, dto);
  }

  @Patch('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'order'])
  async updateOrderStatus(
    @Param('id') id: string,
    @Body() dto: StatusDto
  ) {
    return this.orderService.status(id, dto);
  }

  @Patch(':id/cancel')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  async cancelOrder(
    @Param('id') id: string,
    @CurrentUser() user: Account
  ) {
    return this.orderService.CancelOrder(id, user.id);
  }

  @Get('estimated-delivery/day')
  getEstimatedDeliveryDay() {
    return this.orderService.getCurrentDeliveryDate();
  }

  @Get('admin/track/test/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'order'])
  async getOrder(
    @Param('id') id: string,
  ) {
    return this.orderService.getOrderByIdAdmin(id);
  }

  @Get('admin/track/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.USER, UserRole.RETAILER, UserRole.STAFF)
  async trackOrder(
    @Param('id') id: string,
    @CurrentUser() user: Account
  ) {
    return this.orderService.getOrderById(id);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'))
  async getOrderById(
    @Param('id') id: string,
    @CurrentUser() user: Account
  ) {
    return this.orderService.getOrderById(id)
  }

  @Get('invoice/:id')
  async downloadInvoicePdf(@Param('id') id: string, @Res() res: Response) {
    try {
      const pdfBuffer = await this.orderService.generateInvoicePdf(id);

      if (!pdfBuffer) {
        console.error('PDF buffer is undefined or null');
        throw new NotFoundException('Invoice not found.');
      }

      res.set({
        'Content-Type': 'application/pdf',
        'Content-Disposition': `attachment; filename=invoice_${id}.pdf`,
        'Content-Length': pdfBuffer.length,
      });

      return res.send(pdfBuffer);
    } catch (error) {
      console.error('PDF Generation Error:', error);
      throw new NotFoundException('Order not found or PDF generation failed.');
    }
  }

  @Post(':id/return-request')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, UserRole.RETAILER)
  async requestReturn(
    @Param('id') id: string,
    @Body() dto: ReturnOrderDto,
    @CurrentUser() user: Account
  ) {
    return this.orderService.requestReturnOrder(id, user.id, dto.returnReason);
  }

  @Patch(':id/approve-return')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'order'])
  async approveReturn(
    @Param('id') id: string,
    @CurrentUser() admin: Account
  ) {
    return this.orderService.approveReturnRequest(id, admin.id);
  }

  @Patch(':id/process-refund')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'order'])
  async processRefund(
    @Param('id') id: string,
    @Body() dto: ProcessRefundDto,
    @CurrentUser() admin: Account
  ) {
    return this.orderService.processRefund(id, admin.id, dto.refundAmount);
  }
}

