import { BadRequestException, Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { OrderItem } from 'src/order-item/entities/order-item.entity';
import { Brackets, Repository, QueryRunner, In } from 'typeorm';
import { Order } from './entities/order.entity';
import { Cart } from 'src/cart/entities/cart.entity';
import { CartItem } from 'src/cart-item/entities/cart-item.entity';
import { CreateOrderDto, DirectBuyDto, PaginationDto, StatusDto, VerifyPaymentDto } from './dto/order.dto';
import { AdminCreateOrderDto } from './dto/admin-create-order.dto';
import { Account } from './../account/entities/account.entity';
import { DefaultStatus, NotificationType, OrderStatus, PaymentMethod, PaymentMode, ProductStatus, UserRole, YesNo } from 'src/enum';
import { Product } from 'src/product/entities/product.entity';
import Razorpay from 'razorpay';
import crypto from 'crypto';
import { NotificationsService } from 'src/notifications/notifications.service';
import { deliveryAddress } from 'src/delivery-address/entities/delivery-address.entity';
import { UserDetail } from '../user-details/entities/user-detail.entity';
import { PdfGeneratorUtil } from '../utils/pdf-generator.util';
import { Setting } from 'src/settings/entities/setting.entity';
import { City } from 'src/city/entities/city.entity';

import { CityService } from 'src/city/city.service';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';


@Injectable()
export class OrderService {
  private razorpay = new Razorpay({
    key_id: process.env.RN_RAZORPAY_KEY_ID,
    key_secret: process.env.RN_RAZORPAY_KEY_SECRET,
  });

  constructor(
    @InjectRepository(Order) private orderRepo: Repository<Order>,
    @InjectRepository(OrderItem) private orderItemRepo: Repository<OrderItem>,
    @InjectRepository(Cart) private cartRepo: Repository<Cart>,
    @InjectRepository(CartItem) private cartitemRepo: Repository<CartItem>,
    @InjectRepository(Product) private productRepo: Repository<Product>,
    @InjectRepository(Account) private accountRepo: Repository<Account>,
    @InjectRepository(deliveryAddress) private deliveryAddressRepo: Repository<deliveryAddress>,
    @InjectRepository(UserDetail) private userRepo: Repository<UserDetail>,
    @InjectRepository(Setting) private settingRepo: Repository<Setting>,
    private notificationsService: NotificationsService,
    @InjectRepository(City) private cityRepo: Repository<City>,

    private nodeMailerService: NodeMailerService
  ) { }


 async adminPlaceOrder(dto: AdminCreateOrderDto, adminId: string) {
  const [user, product] = await Promise.all([
    this.accountRepo.findOne({ where: { id: dto.userId } }),
    this.productRepo.findOne({ where: { id: dto.productId } })
  ]);

  if (!user) throw new NotFoundException('User not found');
  if (!product) throw new NotFoundException('Product not found');
  if (product.quantity < dto.quantity) throw new BadRequestException('Not enough stock');

  
  const role = user.roles.includes('RETAILER') ? 'RETAILER' : 'USER';
  const price = role === 'RETAILER' ? product.retailerPrice : product.userPrice;

  const subTotal = price * dto.quantity;
  const gstAmount = subTotal * 0.18; // assuming 18% GST
  const totalAmount = subTotal + gstAmount + dto.deliveryCharge;

  const estimatedDeliveryDate = this.getDeliveryDate();


  const order = await this.orderRepo.save({
    accountId: dto.userId,
    quantity: dto.quantity,
    gstAmount,
    subTotal,
    totalAmount,
    deliveryCharge: dto.deliveryCharge,
    paymentMethod: dto.paymentMethod,
    status: dto.status,
    fullDeliveryAddress: dto.fullDeliveryAddress,
    estimatedDeliveryDate
  });


  await this.orderItemRepo.save({
    orderId: order.id,
    productId: dto.productId,
    quantity: dto.quantity,
    price,
    totalPrice: subTotal
  });


await this.productRepo.decrement({ id: dto.productId }, 'quantity', dto.quantity);


const updatedProduct = await this.productRepo.findOne({ where: { id: dto.productId } });

if (updatedProduct) {
  if (updatedProduct.quantity === 0) {
    await this.productRepo.update(dto.productId, { status: ProductStatus.OUT_OF_STOCK });
  }

  if (updatedProduct.quantity < 5) {
    await this.notifyAdmins(
      'Low Stock Alert',
      `Product "${updatedProduct.name}" is running low on stock. Only ${updatedProduct.quantity} units remaining.`
    );
  }
}


  await this.notificationsService.create({
    title: 'Order Placed by Admin',
    desc: `An order has been placed for you by admin. Order ID: #${order.id.slice(0, 8)}`,
    accountId: dto.userId,
    type: NotificationType.ORDER_PLACED
  });

  
  await this.notifyAdmins('Admin Order Placed', `Admin placed order #${order.id.slice(0, 8)} for customer. Amount: ₹${totalAmount}`);

  return {
    message: 'Order placed successfully by admin',
    orderId: order.id,
    orderDetails: {
      userId: dto.userId,
      productId: dto.productId,
      quantity: dto.quantity,
      price,
      subTotal,
      gstAmount,
      totalAmount,
      status: dto.status
    }
  };
}


  private getDeliveryDate(): string {
    const date = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000);
    const options: Intl.DateTimeFormatOptions = { day: 'numeric', month: 'short', year: 'numeric' };
    return date.toLocaleDateString('en-GB', options).replace(/ /g, ' ');
  }

  private formatDeliveryAddress(address: deliveryAddress): string {
    const parts = [
      address.name,
      address.address,
      address.landMark,
      [address.city, address.state, address.pin].filter(Boolean).map(p => p.trim()).join(' '),
      address.phNumber ? `Phone: ${address.phNumber.trim()}` : ''
    ];
    return parts
      .map(part => part?.trim())
      .filter(part => !!part)
      .join(', ');
  }


  private async calculateTotals(price: number, quantity: number, cityName: string) {
    const settings = await this.settingRepo.findOne({
      where: { status: DefaultStatus.ACTIVE },
    });

    const gstRate = settings ? settings.gstPercentage / 100 : 0.018;

    const subTotal = parseFloat((price * quantity).toFixed(2));
    const gstAmount = parseFloat((subTotal * gstRate).toFixed(2));

    const city = await this.cityRepo.findOne({ where: { name: cityName } });
    if (!city) throw new BadRequestException(`City "${cityName}" not found`);

    const deliveryCharge = city.deliveryCharge ?? 0;

    const totalAmount = parseFloat((subTotal + gstAmount + deliveryCharge).toFixed(2));

    return { subTotal, gstAmount, deliveryCharge, totalAmount };
  }


  async handleCartCheckout(accountId: string, paymentMethod: PaymentMethod, dto: CreateOrderDto) {
    const cart = await this.cartRepo.createQueryBuilder('cart')
      .leftJoinAndSelect('cart.cartItems', 'items')
      .leftJoinAndSelect('items.product', 'product')
      .where('cart.accountId = :accountId', { accountId })
      .getOne();

    if (!cart?.cartItems?.length) {
      throw new NotFoundException('Cart is empty');
    }

    
    const pendingOrder = await this.orderRepo.findOne({
      where: {
        accountId,
        status: OrderStatus.PENDING_PAYMENT,
        createdAt: new Date(Date.now() - 10 * 60 * 1000) 
      }
    });

    if (pendingOrder) {
      throw new BadRequestException('You have a pending payment. Please complete or cancel it first.');
    }

    const deliveryAddress = await this.deliveryAddressRepo.findOne({
      where: { accountId, isDefault: true },
    });

    if (!deliveryAddress) {
      throw new BadRequestException('Please select a delivery address');
    }

    const city = await this.cityRepo.findOne({ where: { name: deliveryAddress.city } });
    if (!city) {
      throw new BadRequestException(`City "${deliveryAddress.city}" not found`);
    }

    if (paymentMethod === PaymentMethod.COD) {
      this.validateCodAvailability(city);
    }

   
    for (const item of cart.cartItems) {
      if (item.product.quantity < item.quantity) {
        throw new BadRequestException(`Insufficient stock for ${item.product.name}. Available: ${item.product.quantity}, Requested: ${item.quantity}`);
      }
    }

    const { subTotal, gstAmount, deliveryCharge, totalAmount } =
      await this.calculateTotals(Number(cart.totalPrice), 1, deliveryAddress.city);

    const estimatedDeliveryDate = this.getDeliveryDate();
    const fullDeliveryAddress = this.formatDeliveryAddress(deliveryAddress);

    if (paymentMethod === PaymentMethod.COD) {
      const order = await this.orderRepo.save({
        accountId,
        subTotal,
        gstAmount,
        deliveryCharge,
        totalAmount,
        status: OrderStatus.PLACED,
        paymentMethod,
        deliveryAddressId: deliveryAddress.id,
        fullDeliveryAddress,
        estimatedDeliveryDate,
      });
      const orderItems = cart.cartItems.map(item => ({
        orderId: order.id,
        productId: item.productId,
        quantity: item.quantity,
        price: item.price,
        totalPrice: item.subTotalPrice,
      }));

      await Promise.all([
        this.orderItemRepo.save(orderItems),
        this.productRepo.createQueryBuilder()
          .update()
          .set({
            quantity: () => `quantity - CASE ${cart.cartItems.map(item =>
              `WHEN id = '${item.productId}' THEN ${item.quantity}`).join(' ')} END`
          })
          .whereInIds(cart.cartItems.map(item => item.productId))
          .execute(),
        
        this.cartitemRepo.delete({ cartId: cart.id }),
        this.cartRepo.update(cart.id, { totalPrice: 0 })
        
      ]);
      
const productIds = cart.cartItems.map(item => item.productId);

const updatedProducts = await this.productRepo.findBy({ id: In(productIds) });

const zeroStockProducts = updatedProducts.filter(product => product.quantity === 0);
const lowStockProducts = updatedProducts.filter(product => product.quantity > 0 && product.quantity < 5);


if (zeroStockProducts.length) {
  await Promise.all(
    zeroStockProducts.map(product =>
      this.productRepo.update(product.id, { status: ProductStatus.OUT_OF_STOCK })
    )
  );
}


if (lowStockProducts.length) {
  await Promise.all(
    lowStockProducts.map(product =>
      this.notifyAdmins(
        'Low Stock Alert',
        `Product "${product.name}" is running low on stock. Only ${product.quantity} units remaining.`
      )
    )
  );
}

      setImmediate(() => {
        this.notifyOrderSuccess(order.id, accountId).catch(error =>
          console.error('Async notification failed:', error)
        );
      });

      return {
        message: 'Order placed via COD',
        orderId: order.id,
      };
    }


    const amountForRazorpay = Math.round(totalAmount * 100);

    const razorpayOrder = await this.razorpay.orders.create({
      amount: amountForRazorpay,
      currency: 'INR',
      receipt: 'receipt_order_' + Date.now(),
      notes: {
        mode: 'cart',
        accountId,
        deliveryAddressId: deliveryAddress.id
      }
    });

    const order = await this.orderRepo.save({
      accountId,
      subTotal,
      gstAmount,
      deliveryCharge,
      totalAmount,
      status: OrderStatus.PENDING_PAYMENT,
      paymentMethod: PaymentMethod.ONLINE,
      razorpayOrderId: razorpayOrder.id,
      deliveryAddressId: deliveryAddress.id,
      fullDeliveryAddress,
      estimatedDeliveryDate
    });

    return {
      razorpayOrderId: razorpayOrder.id,
      amount: amountForRazorpay,
      currency: 'INR',
      key: process.env.RN_RAZORPAY_KEY_ID,
      keySecret: process.env.RN_RAZORPAY_KEY_SECRET,
      message: 'Proceed with Razorpay payment',
      orderId: order.id
    };
  }


  async verifyPaymentSignature(dto: VerifyPaymentDto, accountId: string) {
    if (!process.env.RN_RAZORPAY_KEY_SECRET) {
      throw new BadRequestException('Payment configuration error');
    }

    const expectedSignature = crypto
      .createHmac('sha256', process.env.RN_RAZORPAY_KEY_SECRET)
      .update(`${dto.razorpayOrderId}|${dto.razorpayPaymentId}`)
      .digest('hex');

    if (dto.razorpaySignature !== expectedSignature) {
      throw new BadRequestException('Invalid payment signature');
    }

    const order = await this.orderRepo.findOne({
      where: { razorpayOrderId: dto.razorpayOrderId, accountId },
    });

    if (!order) throw new NotFoundException('Order not found');
    if (order.status !== OrderStatus.PENDING_PAYMENT) {
      throw new BadRequestException('Payment already processed');
    }

    try {
      const payment = await this.razorpay.payments.fetch(dto.razorpayPaymentId);
      if (payment.status !== 'captured' && payment.status !== 'authorized') {
        throw new BadRequestException('Payment not successful');
      }
    } catch (error) {
      throw new BadRequestException(`Payment verification failed: ${error.message}`);
    }

    await this.orderRepo.update(order.id, {
      razorpayPaymentId: dto.razorpayPaymentId,
      status: OrderStatus.PLACED,
    });

    
    if (dto.mode === PaymentMode.DIRECT && dto.productId && dto.quantity) {
      await this.handleDirectOrderItem(order.id, dto.productId, dto.quantity, accountId);
    } else {
      await this.handleCartOrderItems(accountId, order.id);
    }

    setImmediate(async () => {
      try {
        await this.notifyOrderSuccess(order.id, accountId);
        await this.notificationsService.create({
          title: 'Payment Successful',
          desc: `Payment of ₹${order.totalAmount} completed for order #${order.id.slice(0, 8)}`,
          accountId,
          type: NotificationType.PAYMENT_SUCCESS
        });
        await this.notifyAdmins('Payment Received', `Payment of ₹${order.totalAmount} received for order #${order.id.slice(0, 8)}`);
      } catch (error) {
        console.error('Failed to send payment notifications:', error);
      }
    });

    return {
      success: true,
      message: 'Payment verified successfully',
      orderId: order.id,
      paymentId: dto.razorpayPaymentId,
    };
  }
  private async handleDirectOrderItem(orderId: string, productId: string, quantity: number, accountId: string) {
    const [product, account] = await Promise.all([
      this.productRepo.findOne({ where: { id: productId } }),
      this.accountRepo.findOne({ where: { id: accountId } })
    ]);

    if (!product) throw new NotFoundException('Product not found');
    if (!account) throw new NotFoundException('Account not found');
    if (product.quantity < quantity) throw new BadRequestException('Insufficient stock');

    const price = account.roles === UserRole.RETAILER ? product.retailerPrice : product.userPrice;

    await Promise.all([
      this.orderItemRepo.save({
        orderId,
        productId,
        quantity,
        price,
        totalPrice: price * quantity,
      }),
      this.productRepo.decrement({ id: productId }, 'quantity', quantity)
    ]);
const updatedProduct = await this.productRepo.findOne({ where: { id: productId} });

if (updatedProduct) {
  if (updatedProduct.quantity === 0) {
    await this.productRepo.update(productId, { status: ProductStatus.OUT_OF_STOCK });
  }

  if (updatedProduct.quantity < 5) {
    await this.notifyAdmins(
      'Low Stock Alert',
      `Product "${updatedProduct.name}" is running low on stock. Only ${updatedProduct.quantity} units remaining.`
    );
  }
}
  }

  private async handleCartOrderItems(accountId: string, orderId: string) {
    const [cart, account] = await Promise.all([
      this.cartRepo.createQueryBuilder('cart')
        .leftJoinAndSelect('cart.cartItems', 'items')
        .leftJoinAndSelect('items.product', 'product')
        .where('cart.accountId = :accountId', { accountId })
        .getOne(),
      this.accountRepo.findOne({ where: { id: accountId } })
    ]);

    if (!cart || !cart.cartItems.length) throw new NotFoundException('Cart is empty');
    if (!account) throw new NotFoundException('Account not found');

    const orderItems = cart.cartItems.map(item => {
      if (item.product.quantity < item.quantity) {
        throw new BadRequestException(`Insufficient stock for ${item.product.name}`);
      }

      const price = account.roles === UserRole.RETAILER ? item.product.retailerPrice : item.product.userPrice;
      return {
        orderId,
        productId: item.productId,
        quantity: item.quantity,
        price,
        totalPrice: price * item.quantity,
      };
    });

    await Promise.all([
      this.orderItemRepo.save(orderItems),
      
      ...cart.cartItems.map(item =>
        this.productRepo.decrement({ id: item.productId }, 'quantity', item.quantity)
      ),
      this.cartitemRepo.delete({ cartId: cart.id }),
      this.cartRepo.update(cart.id, { totalPrice: 0 })
    ]);
const productIds = cart.cartItems.map(item => item.productId);

const updatedProducts = await this.productRepo.findBy({ id: In(productIds) });

const zeroStockProducts = updatedProducts.filter(product => product.quantity === 0);
const lowStockProducts = updatedProducts.filter(product => product.quantity > 0 && product.quantity < 5);


if (zeroStockProducts.length) {
  await Promise.all(
    zeroStockProducts.map(product =>
      this.productRepo.update(product.id, { status: ProductStatus.OUT_OF_STOCK })
    )
  );
}


if (lowStockProducts.length) {
  await Promise.all(
    lowStockProducts.map(product =>
      this.notifyAdmins(
        'Low Stock Alert',
        `Product "${product.name}" is running low on stock. Only ${product.quantity} units remaining.`
      )
    )
  );
}
 
  }
  async previewBuyNowUser(dto: { productId: string; quantity: number }, accountId: string) {
    const { productId, quantity } = dto;

    const [result, settings] = await Promise.all([
      this.productRepo
        .createQueryBuilder('product')
        .select([
          'product.id',
          'product.name',
          'product.userPrice',
          'product.imageUrl',
        ])
        .where('product.id = :productId', { productId })
        .getOne(),
      this.settingRepo.findOne({ where: { status: DefaultStatus.ACTIVE } })
    ]);

    if (!result) {
      throw new NotFoundException('Product not found');
    }

    const gstRate = settings ? settings.gstPercentage / 100 : 0.018;
    const subTotal = parseFloat((result.userPrice * quantity).toFixed(2));
    const gst = parseFloat((subTotal * gstRate).toFixed(2));
    const deliveryCharge = 0;
    const totalAmount = parseFloat((subTotal + gst + deliveryCharge).toFixed(2));

    return { ...result, quantity, subTotal, gst, deliveryCharge, totalAmount };
  }

  async previewBuyNowRetailer(dto: { productId: string; quantity: number }, accountId: string) {
    const { productId, quantity } = dto;

    const [result, settings] = await Promise.all([
      this.productRepo
        .createQueryBuilder('product')
        .select([
          'product.id',
          'product.name',
          'product.retailerPrice',
          'product.imageUrl',
        ])
        .where('product.id = :productId', { productId })
        .getOne(),
      this.settingRepo.findOne({ where: { status: DefaultStatus.ACTIVE } })
    ]);

    if (!result) {
      throw new NotFoundException('Product not found');
    }

    const gstRate = settings ? settings.gstPercentage / 100 : 0.018;
    const subTotal = parseFloat((result.retailerPrice * quantity).toFixed(2));
    const gst = parseFloat((subTotal * gstRate).toFixed(2));
    const deliveryCharge = 0;
    const totalAmount = parseFloat((subTotal + gst + deliveryCharge).toFixed(2));

    return { ...result, quantity, subTotal, gst, deliveryCharge, totalAmount };
  }

  async userHandleDirectOrder(dto: DirectBuyDto, accountId: string) {


    const [product, address] = await Promise.all([
      this.productRepo
        .createQueryBuilder('product')
        .select(['product.id', 'product.userPrice', 'product.quantity'])
        .where('product.id = :productId', { productId: dto.productId })
        .getOne(),

      this.deliveryAddressRepo.findOne({
        where: { accountId, isDefault: true },
      }),
    ]);

    if (!product) throw new NotFoundException('Product not found');
    if (!address) throw new BadRequestException('Default delivery address not found');
    if (product.quantity < dto.quantity) throw new BadRequestException('Not enough stock');


    const city = await this.cityRepo.findOne({ where: { name: address.city } });
    if (!city) throw new BadRequestException(`City "${address.city}" not found`);



    if (dto.paymentMethod === PaymentMethod.COD) {
      this.validateCodAvailability(city);
    }


    const { subTotal, gstAmount, deliveryCharge, totalAmount } = await this.calculateTotals(
      product.userPrice,
      dto.quantity,
      address.city
    );

    const estimatedDeliveryDate = this.getDeliveryDate();
    const fullDeliveryAddress = this.formatDeliveryAddress(address);

    if (dto.paymentMethod === PaymentMethod.COD) {
    
      const order = await this.orderRepo.save({
        accountId,
        subTotal,
        gstAmount,
        deliveryCharge,
        totalAmount,
        paymentMethod: dto.paymentMethod,
        status: OrderStatus.PLACED,
        deliveryAddressId: address.id,
        fullDeliveryAddress,
        estimatedDeliveryDate,
      });

      await Promise.all([
        this.orderItemRepo.save({
          orderId: order.id,
          productId: dto.productId,
          quantity: dto.quantity,
          price: product.userPrice,
          totalPrice: subTotal,
        }),
        this.productRepo.decrement({ id: dto.productId }, 'quantity', dto.quantity)
      ]);
const updatedProduct = await this.productRepo.findOne({ where: { id: dto.productId } });

if (updatedProduct) {
  if (updatedProduct.quantity === 0) {
    await this.productRepo.update(dto.productId, { status: ProductStatus.OUT_OF_STOCK });
  }

  if (updatedProduct.quantity < 5) {
    await this.notifyAdmins(
      'Low Stock Alert',
      `Product "${updatedProduct.name}" is running low on stock. Only ${updatedProduct.quantity} units remaining.`
    );
  }
}
  
      setImmediate(() => {
        this.notifyOrderSuccess(order.id, accountId).catch(error =>
          console.error('Async notification failed:', error)
        );
      });

      return { message: 'Order placed via COD', orderId: order.id };
    }



    const amountForRazorpay = Math.round(+totalAmount * 100);
    const razorpayOrder = await this.razorpay.orders.create({
      amount: amountForRazorpay,
      currency: 'INR',
      receipt: `receipt_direct_${Date.now()}`,
      notes: {
        mode: 'direct',
        accountId,
        productId: dto.productId,
        quantity: dto.quantity,
        deliveryAddressId: address.id
      }
    });

    const insertResult = await this.orderRepo
      .createQueryBuilder()
      .insert()
      .into('order')
      .values({
        accountId,
        subTotal,
        gstAmount,
        deliveryCharge,
        totalAmount,
        paymentMethod: PaymentMethod.ONLINE,
        status: OrderStatus.PENDING_PAYMENT,
        razorpayOrderId: razorpayOrder.id,
        deliveryAddressId: address.id,
        fullDeliveryAddress,
        estimatedDeliveryDate,
        quantity: dto.quantity
      })
      .execute();

    const orderId = insertResult.identifiers[0].id;

    return {
      razorpayOrderId: razorpayOrder.id,
      amount: amountForRazorpay,
      currency: 'INR',
      key: process.env.RN_RAZORPAY_KEY_ID,
      keySecret: process.env.RN_RAZORPAY_KEY_SECRET,
      message: 'Proceed with Razorpay payment',
      orderId,
      productId: dto.productId,
      quantity: dto.quantity
    };
  }

  async retailerHandleDirectOrder(dto: DirectBuyDto, accountId: string) {

    const [product, address] = await Promise.all([
      this.productRepo
        .createQueryBuilder('product')
        .select(['product.id', 'product.retailerPrice', 'product.quantity'])
        .where('product.id = :productId', { productId: dto.productId })
        .getOne(),
      this.deliveryAddressRepo.findOne({
        where: { accountId, isDefault: true },

      }),
    ]);

    if (!product) throw new NotFoundException('Product not found');
    if (!address) throw new BadRequestException('Delivery address not found');
    if (product.quantity < dto.quantity) throw new BadRequestException('Not enough stock');
    const city = await this.cityRepo.findOne({ where: { name: address.city } });
    if (!city) throw new BadRequestException(`City "${address.city}" not found`);


    if (dto.paymentMethod === PaymentMethod.COD) {
      this.validateCodAvailability(city);
    }


    const { subTotal, gstAmount, deliveryCharge, totalAmount } =
      await this.calculateTotals(product.retailerPrice, dto.quantity, address.city);

    const estimatedDeliveryDate = this.getDeliveryDate();
    const fullDeliveryAddress = this.formatDeliveryAddress(address);

    if (dto.paymentMethod === PaymentMethod.COD) {
     
      const order = await this.orderRepo.save({
        accountId,
        subTotal,
        gstAmount,
        deliveryCharge,
        totalAmount,
        paymentMethod: dto.paymentMethod,
        status: OrderStatus.PLACED,
        deliveryAddressId: address.id,
        fullDeliveryAddress,
        estimatedDeliveryDate,
      });

     
      await Promise.all([
        this.orderItemRepo.save({
          orderId: order.id,
          productId: dto.productId,
          quantity: dto.quantity,
          price: product.retailerPrice,
          totalPrice: subTotal,
        }),
        this.productRepo.decrement({ id: dto.productId }, 'quantity', dto.quantity)
      ]);

 const updatedProduct = await this.productRepo.findOne({ where: { id: dto.productId } });

if (updatedProduct) {
  if (updatedProduct.quantity === 0) {
    await this.productRepo.update(dto.productId, { status: ProductStatus.OUT_OF_STOCK });
  }

  if (updatedProduct.quantity < 5) {
    await this.notifyAdmins(
      'Low Stock Alert',
      `Product "${updatedProduct.name}" is running low on stock. Only ${updatedProduct.quantity} units remaining.`
    );
  }
}
      setImmediate(() => {
        this.notifyOrderSuccess(order.id, accountId).catch(error =>
          console.error('Async notification failed:', error)
        );
      });

      return { message: 'Order placed via COD', orderId: order.id };
    }


    const amountForRazorpay = Math.round(totalAmount * 100);

    const razorpayOrder = await this.razorpay.orders.create({
      amount: amountForRazorpay,
      currency: 'INR',
      receipt: `receipt_direct_${Date.now()}`,
      notes: {
        mode: 'direct',
        accountId,
        productId: dto.productId,
        quantity: dto.quantity,
        deliveryAddressId: address.id,
      },
    });

    const insertResult = await this.orderRepo
      .createQueryBuilder()
      .insert()
      .into('order')
      .values({
        accountId,
        subTotal,
        gstAmount,
        deliveryCharge,
        totalAmount,
        paymentMethod: PaymentMethod.ONLINE,
        status: OrderStatus.PENDING_PAYMENT,
        razorpayOrderId: razorpayOrder.id,
        deliveryAddressId: address.id,
        fullDeliveryAddress,
        estimatedDeliveryDate,
        quantity: dto.quantity,
      })
      .execute();

    const orderId = insertResult.identifiers[0].id;

    return {
      razorpayOrderId: razorpayOrder.id,
      amount: amountForRazorpay,
      currency: 'INR',
      key: process.env.RN_RAZORPAY_KEY_ID,
      keySecret: process.env.RN_RAZORPAY_KEY_SECRET,
      message: 'Proceed with Razorpay payment',
      orderId,
      productId: dto.productId,
      quantity: dto.quantity,
    };
  }


  async findAll(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.orderRepo.createQueryBuilder('order')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product')
      .leftJoinAndSelect('order.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'order.id',
        'order.accountId',
        'order.subTotal',
        'order.gstAmount',
        'order.totalAmount',
        'order.deliveryCharge',
        'order.paymentMethod',
        'order.status',
        'order.createdAt',
        'order.updatedAt',
        'order.estimatedDeliveryDate',
        'order.fullDeliveryAddress',
        'orderItems.id',
        'orderItems.quantity',
        'orderItems.price',
        'orderItems.totalPrice',
        'product.id',
        'product.name',
        'account.id',
        'account.phoneNumber',
        'product.imageUrl',
        'userDetail.id',
        'userDetail.name',
        'userDetail.email'
      ]);
    if (keyword) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.id LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.rating LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.status LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.id LIKE :keyword', { keyword: `%${keyword}%` });
        }),
      );
    }
    if (dto.status) {
      query.andWhere('order.status = :status', { status: dto.status });
    }
    if (dto.date) {
      const startDate = new Date(dto.date);
      startDate.setHours(0, 0, 0, 0);
      const endDate = new Date(startDate);
      endDate.setDate(endDate.getDate() + 1);
      query.andWhere('order.createdAt >= :startDate AND order.createdAt < :endDate', {
        startDate,
        endDate
      });
    }
    const [result, total] = await query
      .orderBy('order.createdAt', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }
  async getOrdersOfUserByAdmin(accountId: string, dto: PaginationDto) {
    const keyword = dto.keyword || '';

    const query = this.orderRepo.createQueryBuilder('order')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product')
      .leftJoinAndSelect('order.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('product.ratingFeedback', 'ratingFeedback', 'ratingFeedback.accountId = :accountId', { accountId })
      .where('order.accountId = :accountId', { accountId })
      .select([
        'order.id',
        'order.accountId',
        'order.subTotal',
        'order.gstAmount',
        'order.totalAmount',
        'order.paymentMethod',

        'order.status',
        'order.createdAt',
        'order.updatedAt',
        'order.estimatedDeliveryDate',
        'order.fullDeliveryAddress',
        'orderItems',
        'product',
        'account',
        'userDetail',
        'ratingFeedback'
      ]);

    if (keyword) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.id LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.rating LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('userDetail.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('ratingFeedback.feedback LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.status LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.id LIKE :keyword', { keyword: `%${keyword}%` });
        }),
      );
    }
    if (dto.status) {
      query.andWhere('order.status = :status', { status: dto.status });
    }
    if (dto.date) {
      const startDate = new Date(dto.date);
      startDate.setHours(0, 0, 0, 0);
      const endDate = new Date(startDate);
      endDate.setDate(endDate.getDate() + 1);
      query.andWhere('order.createdAt >= :startDate AND order.createdAt < :endDate', {
        startDate,
        endDate
      });
    }

    const [result, total] = await query
      .orderBy('order.updatedAt', 'DESC')
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { total, result };
  }


  async getOrders(accountId: string, dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.orderRepo.createQueryBuilder('order')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product')
      .leftJoinAndSelect('product.ratingFeedback', 'ratingFeedback', 'ratingFeedback.accountId = :accountId', { accountId })
      .where('order.accountId = :accountId AND order.status != :status', { accountId, status: OrderStatus.PENDING_PAYMENT })
      .select([
        'order.id',
        'order.subTotal',
        'order.gstAmount',
        'order.totalAmount',
        'order.createdAt',
        'order.updatedAt',
        'order.status',
        'order.estimatedDeliveryDate',
        'order.fullDeliveryAddress',
        'orderItems.id',
        'orderItems.quantity',
        'product.id',
        'product.name',
        'product.imageUrl',
        'product.rating',
        'ratingFeedback.id',
        'ratingFeedback.rating',
        'ratingFeedback.feedback',
        'ratingFeedback.status',
      ]);

    if (keyword) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.id LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.rating LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('ratingFeedback.feedback LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.status LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('order.id LIKE :keyword', { keyword: `%${keyword}%` });
        }),
      );
    }

    const [result, total] = await query
      .orderBy('order.updatedAt', 'DESC')
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { total, result };
  }

  async status(id: string, dto: StatusDto) {
    const order = await this.orderRepo.findOne({
      where: { id },
      relations: ['orderItems', 'orderItems.product']
    });
    if (!order) throw new NotFoundException('Order not found!');

    await this.orderRepo.update(id, { status: dto.status });

    await this.notificationsService.create({
      title: 'Order Status Updated',
      desc: `Your order #${id.substring(0, 8)} status has been updated to ${dto.status}.`,
      accountId: order.accountId,
      type: NotificationType.ORDER_STATUS_CHANGE
    });


    await this.notifyAdmins('Order Status Changed', `Order #${id.slice(0, 8)} status updated to ${dto.status}`);

    return { message: `Order status updated to ${dto.status}`, orderId: id };
  }

  async CancelOrder(id: string, accountId: string) {
    const order = await this.orderRepo.findOne({
      where: { id, accountId },
      relations: ['orderItems', 'orderItems.product']
    });
    if (!order) throw new NotFoundException('Order not found!');
    if ([OrderStatus.CANCELED, OrderStatus.DELIVERED, OrderStatus.COMPLETED].includes(order.status)) {
      throw new BadRequestException('Cannot cancel this order');
    }
    await this.orderRepo.update(id, { status: OrderStatus.CANCELED });

    await this.notificationsService.create({
      title: 'Order Cancelled',
      desc: `Your order #${id.substring(0, 8)} has been cancelled`,
      accountId,
      type: NotificationType.ORDER_CANCELLED
    });


    await this.notifyAdmins('Order Cancelled', `Customer cancelled order #${id.slice(0, 8)}. Refund may be required.`);

    return { message: 'Order cancelled successfully' };
  }

  async getOrderById(id: string) {
    const order = await this.orderRepo.createQueryBuilder('order')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product')
      .select([
        'order.id',
        'order.accountId',
        'order.subTotal',
        'order.gstAmount',
        'order.totalAmount',
        'order.paymentMethod',
        'order.status',
        'order.fullDeliveryAddress',
        'order.createdAt',
        'order.updatedAt',
        'order.estimatedDeliveryDate',
        'orderItems',
        'product.id',
        'product.name',
        'product.imageUrl',
      ])
      .where('order.id = :id', { id })
      .getOne();

    if (!order) throw new NotFoundException('Order not found');

    const userDetail = await this.userRepo.findOne({
      where: { accountId: order.accountId },
      select: ['id', 'name', 'email']
    });

    return { order, user: userDetail, estimatedDeliveryDate: order.estimatedDeliveryDate };
  }

  async getOrderByIdAdmin(id: string) {
    const order = await this.orderRepo.createQueryBuilder('order')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product')
      .select([
        'order.id',
        'order.accountId',
        'order.subTotal',
        'order.gstAmount',
        'order.totalAmount',
        'order.paymentMethod',
        'order.status',
        'order.razorpayOrderId',
        'order.createdAt',
        'order.updatedAt',
        'order.estimatedDeliveryDate',
        'order.fullDeliveryAddress',
        'order.deliveryCharge',

        'orderItems.id',
        'orderItems.quantity',
        'orderItems.price',
        'orderItems.totalPrice',

        'product.id',
        'product.name',
        'product.description',
        'product.imageUrl',
        'product.price',
        'product.rating',

      ])
      .where('order.id = :id', { id })
      .getOne();

    if (!order) throw new NotFoundException('Order not found');

    const userDetail = await this.userRepo.findOne({
      where: { accountId: order.accountId },
      select: ['id', 'name', 'email'],
    });

    return { order, user: userDetail };
  }


  async trackOrderForUser(id: string, accountId: string) {
    const order = await this.orderRepo.createQueryBuilder('order')
      .where('order.id = :id AND order.accountId = :accountId', { id, accountId })
      .select([
        'order.id',
        'order.status',
        'order.createdAt',
        'order.updatedAt',
        'order.estimatedDeliveryDate',
        'order.fullDeliveryAddress',
      ])
      .getOne();

    if (!order) throw new NotFoundException('Order not found');

    return order;
  }

  getCurrentDeliveryDate() {
    return { estimatedDeliveryDate: this.getDeliveryDate() };
  }

  private async notifyOrderSuccess(orderId: string, accountId: string) {
    await this.notificationsService.create({
      title: 'Order Placed Successfully',
      desc: `Your order #${orderId.slice(0, 8)} has been placed successfully.`,
      accountId,
      type: NotificationType.ORDER_PLACED,
    });
    await this.notifyAdmins('New Order Received', `Order #${orderId.slice(0, 8)} placed by customer. Requires processing.`);
    await this.sendOrderEmails(orderId, accountId);
  }

  async requestReturnOrder(orderId: string, accountId: string, returnReason: string) {
    const order = await this.orderRepo.findOne({
      where: { id: orderId, accountId },
      relations: ['orderItems', 'orderItems.product']
    });
    if (!order) throw new NotFoundException('Order not found!');

    if (order.status !== OrderStatus.DELIVERED && order.status !== OrderStatus.COMPLETED) {
      throw new BadRequestException('Only delivered or completed orders can be returned');
    }

    await this.orderRepo.update(orderId, { status: OrderStatus.RETURN_REQUESTED });

    await this.notificationsService.create({
      title: 'Return Request Received',
      desc: `Your return request for order #${orderId.slice(0, 8)} has been received.`,
      accountId,
      type: NotificationType.RETURN_REQUESTED
    });

    await this.notifyAdmins('Return Request', `Customer requested return for order #${orderId.slice(0, 8)}. Reason: ${returnReason}`);

    return { message: 'Return request submitted successfully' };
  }

  async approveReturnRequest(orderId: string, adminId: string) {
    const order = await this.orderRepo.findOne({
      where: { id: orderId },
      relations: ['orderItems', 'orderItems.product']
    });
    if (!order) throw new NotFoundException('Order not found!');

    if (order.status !== OrderStatus.RETURN_REQUESTED) {
      throw new BadRequestException('Order is not in return requested state');
    }

    await this.orderRepo.update(orderId, { status: OrderStatus.RETURN_APPROVED });

    await this.notificationsService.create({
      title: 'Return Request Approved',
      desc: `Your return request for order #${orderId.slice(0, 8)} has been approved.`,
      accountId: order.accountId,
      type: NotificationType.RETURN_APPROVED
    });


    return { message: 'Return request approved successfully' };
  }

  async processRefund(orderId: string, adminId: string, refundAmount: number) {
    const order = await this.orderRepo.findOne({
      where: { id: orderId },
      relations: ['orderItems', 'orderItems.product']
    });
    if (!order) throw new NotFoundException('Order not found!');

    if (order.status !== OrderStatus.RETURN_APPROVED) {
      throw new BadRequestException('Return must be approved before processing refund');
    }

    await this.orderRepo.update(orderId, { status: OrderStatus.REFUNDED });

    await this.notificationsService.create({
      title: 'Refund Processed',
      desc: `Your refund of ₹${refundAmount} for order #${orderId.slice(0, 8)} has been processed.`,
      accountId: order.accountId,
      type: NotificationType.REFUND_PROCESSED
    });

    await this.notifyAdmins('Refund Processed', `Refund of ₹${refundAmount} processed for order #${orderId.slice(0, 8)}`);

    return { message: 'Refund processed successfully' };
  }

  private async notifyAdmins(title: string, desc: string) {
    const admins = await this.accountRepo.find({ where: { roles: UserRole.ADMIN } });
    for (const admin of admins) {
      await this.notificationsService.create({
        title,
        desc,
        accountId: admin.id,
        type: NotificationType.SYSTEM,
      });
    }
  }



  async generateInvoicePdf(id: string) {
    const orderData = await this.getOrderByIdAdmin(id);
    const settings = await this.settingRepo.findOne({ where: { status: DefaultStatus.ACTIVE } });

    if (!orderData.order.fullDeliveryAddress && orderData.order.deliveryAddressId) {
      const address = await this.deliveryAddressRepo.findOne({ where: { id: orderData.order.deliveryAddressId } });
      if (address) {
        orderData.order.fullDeliveryAddress = this.formatDeliveryAddress(address);
      } else {
        orderData.order.fullDeliveryAddress = 'Address information not available';
      }
    } else if (!orderData.order.fullDeliveryAddress) {
      orderData.order.fullDeliveryAddress = 'Address information not available';
    }

    return PdfGeneratorUtil.generateInvoice(orderData, settings);
  }
  private validateCodAvailability(city: City): void {
    if (!city) {
      throw new BadRequestException('City not found for delivery address');
    }

    if (city.isCodAvailable !== YesNo.YES) {
      throw new BadRequestException(`Cash on Delivery is not available for ${city.name}`);
    }
  }

  private async sendOrderEmails(orderId: string, accountId: string) {
    try {
      const orderData = await this.getOrderByIdAdmin(orderId);
      const userEmail = orderData.user?.email;
      const userName = orderData.user?.name || 'Valued Customer';

      if (!userEmail) {
        return;
      }

     
      await this.nodeMailerService.sendOrderConfirmation(userEmail, orderData.order, userName);

      
      setImmediate(() => {
        this.nodeMailerService.sendNewOrderNotificationToAdmin(orderData.order, userName)
          .catch(error => console.error('Admin email failed:', error));
      });

    } catch (error) {
      console.error('Failed to send order emails:', error);
    }
  }

}