import { Body, Controller, Get, Param, Patch, Post, UseGuards } from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { ApiTags } from '@nestjs/swagger';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PageDto } from './dto/page.dto';
import { PagesService } from './pages.service';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';

@ApiTags('Pages')
@Controller('pages')
export class PagesController {
  constructor(private readonly pagesService: PagesService) { }

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'page'])
  create(@Body() dto: PageDto) {
    return this.pagesService.create(dto);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'))
  findAll() {
    return this.pagesService.findAll();
  }

  @Get(':id')
  findOne(@Param('id') id: number) {
    const pageId = Number(id);
    if (isNaN(pageId)) {
      throw new Error('Invalid page ID. Must be a number.');
    }
    return this.pagesService.findOne(pageId);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
    @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'page'])
  update(@Param('id') id: string, @Body() updatePageDto: PageDto) {
    const pageId = Number(id);
    if (isNaN(pageId)) {
      throw new Error('Invalid page ID. Must be a number.');
    }
    return this.pagesService.update(pageId, updatePageDto);
  }
}
