import { Controller, Get, Post, Body, Patch, Param, Delete, MaxFileSizeValidator, ParseFilePipe, UploadedFiles, UseGuards, UseInterceptors, FileTypeValidator, UploadedFile, Put, BadRequestException } from '@nestjs/common';
import { ProductImagesService } from './product-images.service';
import { CreateProductImageDto } from './dto/create-product-image.dto';
import { UpdateProductImageDto } from './dto/update-product-image.dto';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { AuthGuard } from '@nestjs/passport';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionAction, UserRole } from 'src/enum';
import { FileInterceptor, FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';


@Controller('product-images')
export class ProductImagesController {
  constructor(private readonly productImagesService: ProductImagesService) {}



@Put('updateImages/:id')
@UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
    @CheckPermissions([PermissionAction.UPDATE, 'product_images'])
@UseInterceptors(
  FileInterceptor('file', {
    storage: diskStorage({
      destination: './uploads/products/images',
      filename: (req, file, callback) => {
        const randomName = Array(32)
          .fill(null)
          .map(() => Math.round(Math.random() * 16).toString(16))
          .join('');
        return callback(null, `${randomName}${extname(file.originalname)}`);
      },
    }),
  }),
)
async updateImage(
  @Param('id') id: string,
  @CurrentUser() user: Account,
  @UploadedFile(
    new ParseFilePipe({
      validators: [
        new FileTypeValidator({ fileType: '.(png|jpeg|jpg|mp4)' }),
        new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
      ],
    }),
  )
  file: Express.Multer.File,
) {
  const fileData = await this.productImagesService.findOne(id);
  return this.productImagesService.updateImage(file.path, fileData,user.id);
}

@Delete(':id')
@UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'product_images'])
remove(@Param('id') id: string) {
  return this.productImagesService.remove(id);
}
}
