import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Query,
  UseGuards,
  UsePipes,
  ValidationPipe,
  UploadedFile,
  ParseFilePipe,
  FileTypeValidator,
  MaxFileSizeValidator,
  Put,
  UseInterceptors,
  UploadedFiles,
  BadRequestException,
} from '@nestjs/common';
import { ProductService } from './product.service';
import { CreateProductDto } from './dto/create-product.dto';
import { UpdateProductDto, UpdateProductStatusDto, } from './dto/update-product.dto';
import { ProductPaginationDto } from './dto/product-pagination.dto';
import { SearchProductDto } from './dto/search-product.dto';
import { JwtAuthGuard } from '../auth/guards/jwt.guard';
import { PermissionAction, UserRole } from '../enum';
import { CurrentUser } from '../auth/decorators/current-user.decorator';
import { Account } from '../account/entities/account.entity';
import { Roles } from '../auth/decorators/roles.decorator';
import { RolesGuard } from '../auth/guards/roles.guard';
import { AuthGuard } from '@nestjs/passport';
import { FileInterceptor, FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { ProductImage } from 'src/product-images/entities/product-image.entity';
import { ProductImagesService } from 'src/product-images/product-images.service';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';

@Controller('products')
export class ProductController {
  constructor(
    private readonly productService: ProductService,
    private readonly ProductImageService: ProductImagesService,
  ) { }

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'product'])
  async create(
    @Body() createProductDto: CreateProductDto,
  ) {
    return this.productService.create(createProductDto);
  }

  @Get('admin')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'product'])
  async findAll(
    @Query() paginationDto: ProductPaginationDto,
  ) {
    console.log()
    return this.productService.findAll(paginationDto);
  }

  @Get('list/user')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async findAllByUser(
    @Query() paginationDto: ProductPaginationDto,
    @CurrentUser() user: Account
  ) {
    return this.productService.findAllForUser(paginationDto, user.id);
  }

  @Get('retailer')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async findAllByReailer(
    @Query() paginationDto: ProductPaginationDto,
    @CurrentUser() user: Account
  ) {
    return this.productService.findAllForRetailer(paginationDto, user.id);
  }
  @Get('user/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async findForUser(
    @Param('id') id: string,
    @CurrentUser() user: Account
  ) {
    return this.productService.findOneForUser(id, user.id);
  }

  @Get('user/:id/similar')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async findSimilarForUser(
    @Param('id') id: string,
    @CurrentUser() user: Account,
    @Query() dto: ProductPaginationDto
  ) {
    return this.productService.findSimilarProductsForUser(id, user.id, dto);
  }

  @Get('retailer/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async findUser(
    @Param('id') id: string,
    @CurrentUser() user: Account
  ) {
    return this.productService.findOneForRetailer(id, user.id);
  }

  @Get('retailer/:id/similar')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER)
  async findSimilarForRetailer(
    @Param('id') id: string,
    @CurrentUser() user: Account,
    @Query() dto: ProductPaginationDto
  ) {
    return this.productService.findSimilarProductsForRetailer(id, user.id, dto);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'product'])
  async update(
    @Param('id') id: string,
    @Body() updateProductDto: UpdateProductDto,
  ) {
    return this.productService.update(id, updateProductDto);
  }

  @Patch('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'product'])
  async updateStatus(
    @Param('id') id: string,
    @Body() statusDto: UpdateProductStatusDto,
  ) {
    return this.productService.updateStatus(id, statusDto);
  }

  @Patch('quantity/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'product'])
  async updateQuantity(
    @Param('id') id: string,
    @Body('quantity') quantity: number,
  ) {
    return this.productService.updateQuantity(id, quantity);
  }

  @Put('uploadImage/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'product'])
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/Product',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async crzyImage(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          // new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.productService.find(id);
    return this.productService.image(file.path, fileData);
  }

  @Put('uploadImages/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'product'])
  @UseInterceptors(
    FilesInterceptor('files', 10, {
      storage: diskStorage({
        destination: './uploads/products/images',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async uploadMultipleImages(
    @Param('id') productId: string,
    @UploadedFiles(
      new ParseFilePipe({
        validators: [
          // new FileTypeValidator({ fileType: '.(png|jpeg|jpg|mp4)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    files: Express.Multer.File[],
  ) {
    if (!files || files.length === 0) {
      throw new BadRequestException('Files are required!');
    }

    const product = await this.productService.find(productId);
    return this.productService.saveMultipleImages(product, files);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'product'])
  async remove(@Param('id') id: string) {
    return this.productService.remove(id);
  }
}
