import {
  Injectable,
  NotFoundException,
  BadRequestException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, MoreThan, LessThan, In, Brackets } from 'typeorm';
import { Product } from './entities/product.entity';
import { CreateProductDto } from './dto/create-product.dto';
import { UpdateProductDto, UpdateProductStatusDto } from './dto/update-product.dto';
import { ProductPaginationDto } from './dto/product-pagination.dto';
import { ProductStatus } from '../enum';
import { Brand } from '../brand/entities/brand.entity';
import { Category } from '../category/entities/category.entity';
import { ProductImage } from 'src/product-images/entities/product-image.entity';

@Injectable()
export class ProductService {
  constructor(
    @InjectRepository(Product)
    private readonly productRepository: Repository<Product>,
    @InjectRepository(Brand)
    private readonly brandRepository: Repository<Brand>,
    @InjectRepository(Category)
    private readonly categoryRepository: Repository<Category>,
    @InjectRepository(ProductImage)
    private readonly productImageRepository: Repository<ProductImage>,
  ) { }

 async create(dto: CreateProductDto) {
    const brand = await this.brandRepository.findOne({ where: { id: dto.brandId } });
    if (!brand) {
      throw new NotFoundException('Brand not found.');
    }
    const category = await this.categoryRepository.findOne({ where: { id: dto.categoryId } });
    if (!category) {
      throw new NotFoundException('Category with not found.');
    }
    const newProduct = Object.create(dto);
    
    if (!dto.quantity || dto.quantity === 0) {
      newProduct.status = ProductStatus.OUT_OF_STOCK;
    }
    
    return await this.productRepository.save(newProduct);
  } 
  async findAll(dto: ProductPaginationDto) {
    const queryBuilder = this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.userPrice',
        'product.retailerPrice',
        'product.quantity',
        'product.status',
        'product.userDiscount',
        'product.retailerDiscount',
        'product.rating',
        'product.color',
        'product.reviewCount',
        'product.highlights',
        'product.replacement',
        'product.imageUrl',
        'product.createdAt',
        'product.updatedAt',
        'product.isCrazyDeal',
        'product.isHotDeals',
        'category.id',
        'category.name',
        'brand.id',
        'brand.name',
        'brand.image',

      ]);
    if (dto.keyword) {
      queryBuilder.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :name OR product.id LIKE :name OR product.description LIKE :name OR brand.name LIKE:name OR category.name LIKE :name OR product.highlights LIKE :name',
            { name: `%${dto.keyword}%` })
        }),
      );
    }

    if (dto.categoryId && dto.categoryId.length > 0) {
      queryBuilder.andWhere('product.categoryId = :categoryId', { categoryId: dto.categoryId });
    }

    if (dto.brandId && dto.brandId.length > 0) {
      queryBuilder.andWhere('product.brandId = :brandId', { brandId: dto.brandId });
    }
    if (dto.productId && dto.productId.length > 0) {
      queryBuilder.andWhere('product.id = :productId', { productId: dto.productId });
    }

    if (dto.status) {
      queryBuilder.andWhere('product.status = :status', { status: dto.status });
    }

    if (dto.ratingAbove4) {
      queryBuilder.andWhere('product.rating > :ratingAbove', { ratingAbove: 4.0 });
    }

    if (dto.discountAbove15) {
      queryBuilder.andWhere('product.discountPercentage > :discountAbove', { discountAbove: 15 });
    }

    if (dto.inStockOnly) {
      queryBuilder.andWhere('product.status = :status', { status: ProductStatus.IN_STOCK });
    }

    if (dto.priceLowToHigh) {
      queryBuilder.addOrderBy('product.userPrice', 'ASC');
    }
    if (dto.priceHighToLow) {
      queryBuilder.addOrderBy('product.userPrice', 'DESC');
    }
    const [result, total] = await queryBuilder
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'product.name': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }
  async findAllForUser(dto: ProductPaginationDto, accountId: string) {
    const keyword = dto.keyword || '';
    const queryBuilder = this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.productImage', 'productImage')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.userPrice',
        'product.quantity',
        'product.status',
        'product.userDiscount',
        'product.rating',
        'product.reviewCount',
        'product.highlights',
        'product.replacement',
        'product.imageUrl',
        'product.createdAt',
        'product.updatedAt',
        'product.isCrazyDeal',
        'product.isHotDeals',
        'category.id',
        'category.name',
        'brand.id',
        'brand.name',
        'brand.image',
        'wishlist.id',
      ])
      .where('product.status != :inactiveStatus', { inactiveStatus: 'inactive' });
    if (keyword) {
      queryBuilder.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.description LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('brand.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('category.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.highlights LIKE :keyword', { keyword: `%${keyword}%` });
        }),
      );
    }
    if (dto.categoryId?.length > 0) {
      queryBuilder.andWhere('product.categoryId = :categoryId', { categoryId: dto.categoryId });
    }

    if (dto.brandId?.length > 0) {
      queryBuilder.andWhere('product.brandId = :brandId', { brandId: dto.brandId });
    }

    if (dto.status) {
      queryBuilder.andWhere('product.status = :status', { status: dto.status });
    }

    if (dto.ratingAbove4) {
      queryBuilder.andWhere('product.rating > :ratingAbove', { ratingAbove: 4.0 });
    }

    if (dto.discountAbove15) {
      queryBuilder.andWhere('product.userDiscount > :discountAbove', { discountAbove: 15 });
    }

    if (dto.inStockOnly) {
      queryBuilder.andWhere('product.status = :inStock', { inStock: ProductStatus.IN_STOCK });
    }

    if (dto.hotDeals) {
      queryBuilder.andWhere('product.isHotDeals = true');
    }

    if (dto.crezyDeal) {
      queryBuilder.andWhere('product.isCrazyDeal = true');
    }

    if (dto.priceLowToHigh) {
      queryBuilder.addOrderBy('product.userPrice', 'ASC');
    } else if (dto.priceHighToLow) {
      queryBuilder.addOrderBy('product.userPrice', 'DESC');
    }

    queryBuilder.addOrderBy('product.name', 'ASC');

    const [result, total] = await queryBuilder
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async findAllForRetailer(dto: ProductPaginationDto, accountId: string) {
    const keyword = dto.keyword || '';
    const queryBuilder = this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.retailerPrice',
        'product.quantity',
        'product.status',
        'product.retailerDiscount',
        'product.rating',
        'product.reviewCount',
        'product.highlights',
        'product.replacement',
        'product.imageUrl',
        'product.createdAt',
        'product.updatedAt',
        'product.isCrazyDeal',
        'product.isHotDeals',
        'category.id',
        'category.name',
        'brand.id',
        'brand.name',
        'brand.image',
        'wishlist.id',
      ])
      .where('product.status != :inactiveStatus', { inactiveStatus: ProductStatus.INACTIVE });
    if (keyword) {
      queryBuilder.andWhere(
        new Brackets((qb) => {
          qb.where('product.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.description LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('brand.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('category.name LIKE :keyword', { keyword: `%${keyword}%` })
            .orWhere('product.highlights LIKE :keyword', { keyword: `%${keyword}%` });
        }),
      );
    }
    if (dto.categoryId?.length > 0) {
      queryBuilder.andWhere('product.categoryId = :categoryId', { categoryId: dto.categoryId });
    }

    if (dto.brandId?.length > 0) {
      queryBuilder.andWhere('product.brandId = :brandId', { brandId: dto.brandId });
    }

    if (dto.status) {
      queryBuilder.andWhere('product.status = :status', { status: dto.status });
    }

    if (dto.ratingAbove4) {
      queryBuilder.andWhere('product.rating > :ratingAbove', { ratingAbove: 4.0 });
    }

    if (dto.discountAbove15) {
      queryBuilder.andWhere('product.retailerDiscount > :discountAbove', { discountAbove: 15 });
    }

    if (dto.inStockOnly) {
      queryBuilder.andWhere('product.status = :status', { status: ProductStatus.IN_STOCK });
    }

    if (dto.hotDeals) {
      queryBuilder.andWhere('product.isHotDeals = true');
    }

    if (dto.crezyDeal) {
      queryBuilder.andWhere('product.isCrazyDeal = true');
    }

    if (dto.priceLowToHigh) {
      queryBuilder.addOrderBy('product.retailerPrice', 'ASC');
    } else if (dto.priceHighToLow) {
      queryBuilder.addOrderBy('product.retailerPrice', 'DESC');
    } else {
      queryBuilder.addOrderBy('product.name', 'ASC');
    }

    const [result, total] = await queryBuilder
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }
  async findOneForUser(id: string, accountId: string) {
    const product = await this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.productImage', 'productImage')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.userPrice',
        'product.quantity',
        'product.status',
        'product.userDiscount',
        'product.rating',
        'product.replacement',
        'product.reviewCount',
        'product.highlights',
        'product.imageUrl',
        'product.imagePath',
        'product.createdAt',
        'product.updatedAt',
        'category.id',
        'category.name',
        'brand.id',
        'brand.name',
        'brand.image',
        'productImage.image',
        'productImage.imagePath',
        'wishlist.id',

      ])
      .where('product.id = :id', { id })
      .getOne();

    return product;
  }
  async findOneForRetailer(id: string, accountId: string) {
    const product = await this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.productImage', 'productImage')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.retailerPrice',
        'product.quantity',
        'product.status',
        'product.retailerDiscount',
        'product.rating',
        'product.replacement',
        'product.reviewCount',
        'product.highlights',
        'product.imageUrl',
        'product.imagePath',
        'product.createdAt',
        'product.updatedAt',
        'category.id',
        'category.name',
        'brand.id',
        'brand.name',
        'brand.image',
        'productImage.image',
        'productImage.imagePath',
        'wishlist.id',
      ])
      .where('product.id = :id', { id })
      .getOne();
    return product;
  }
  async findSimilarProductsForUser(productId: string, accountId: string, dto: ProductPaginationDto) {
    const originalProduct = await this.productRepository.findOne({
      where: { id: productId },
      relations: ['brand', 'category'],
    });

    if (!originalProduct) {
      throw new NotFoundException(`Product with ID "${productId}" not found.`);
    }
    const keyword = originalProduct.name.toLowerCase();
    const queryBuilder = this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.userPrice',
        'product.status',
        'product.userDiscount',
        'product.rating',
        'product.imageUrl',
        'brand.id',
        'brand.name',
        'category.id',
        'category.name',
        'wishlist.id',
      ])
      .where('product.id != :productId', { productId })
      .andWhere(
        new Brackets((qb) => {
          qb.where('product.categoryId = :categoryId', { categoryId: originalProduct.category.id })
            .orWhere('LOWER(product.name) LIKE :keyword', { keyword: `%${keyword}%` })
        }),
      );

    const [result, total] = await queryBuilder
      .orderBy('product.rating', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();
    return { result, total }
  }
  async findSimilarProductsForRetailer(productId: string, accountId: string, dto: ProductPaginationDto) {
    const originalProduct = await this.productRepository.findOne({
      where: { id: productId },
      relations: ['brand', 'category'],
    });
    if (!originalProduct) {
      throw new NotFoundException(`Product with ID "${productId}" not found.`);
    }
    const keyword = originalProduct.name.toLowerCase();
    const queryBuilder = this.productRepository
      .createQueryBuilder('product')
      .leftJoinAndSelect('product.category', 'category')
      .leftJoinAndSelect('product.brand', 'brand')
      .leftJoinAndSelect('product.wishlist', 'wishlist', 'wishlist.accountId = :accountId', { accountId })
      .select([
        'product.id',
        'product.name',
        'product.description',
        'product.price',
        'product.retailerPrice',
        'product.status',
        'product.retailerDiscount',
        'product.rating',
        'product.imageUrl',
        'brand.id',
        'brand.name',
        'category.id',
        'category.name',
        'wishlist.id',
      ])
      .where('product.id != :productId', { productId })
      .andWhere(
        new Brackets((qb) => {
          qb.where('product.categoryId = :categoryId', { categoryId: originalProduct.category.id })
            .orWhere('LOWER(product.name) LIKE :keyword', { keyword: `%${keyword}%` })
        }),
      );
    const [result, total] = await queryBuilder
      .orderBy('product.rating', 'DESC')
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();
    return { result, total }
  }
  async update(id: string, dto: UpdateProductDto) {
    const product = await this.productRepository.findOne({ where: { id } });
    if (!product) throw new NotFoundException('Product not found');

    if (dto.brandId) {
      const brandExists = await this.brandRepository.findOne({ where: { id: dto.brandId } });
      if (!brandExists) throw new NotFoundException('Brand not found.');
    }
    if (dto.categoryId) {
      const categoryExists = await this.categoryRepository.findOne({ where: { id: dto.categoryId } });
      if (!categoryExists) throw new NotFoundException('Category not found.');
    }

    Object.assign(product, dto);
    return await this.productRepository.save(product);
  }

  async updateStatus(id: string, dto: UpdateProductStatusDto) {
    const product = await this.productRepository.findOne({ where: { id: id } });
    if (!product) {
      throw new NotFoundException('Product not found');
    }

    product.status = dto.status;
    return await this.productRepository.save(product);
  }

  async updateQuantity(id: string, quantity: number) {
    const product = await this.productRepository.findOne({ where: { id } });
    if (!product) throw new NotFoundException('Product not found');

    if (quantity < 0 && Math.abs(quantity) > product.quantity) {
      throw new BadRequestException('Insufficient product quantity');
    }

    product.quantity += quantity;
    product.status =
      product.quantity > 0
        ? ProductStatus.IN_STOCK
        : ProductStatus.OUT_OF_STOCK;

    return await this.productRepository.save(product);
  }


  async find(id: string) {
    const product = await this.productRepository.findOne({ where: { id } });
    if (!product) {
      throw new NotFoundException('Product not found!');
    }
    return product;
  }

  async image(image: string, result: Product) {
    const obj = Object.assign(result, {
      imageUrl: process.env.RN_CDN_LINK + image,
      imagePath: image,
    });
    return await this.productRepository.save(obj);
  }

  async saveMultipleImages(product: Product, files: Express.Multer.File[]) {
    const productImages = files.map((file) => {
      return this.productImageRepository.create({
        product: product,
        productId: product.id,
        image: `${process.env.RN_CDN_LINK}${file.path}`,
        imagePath: file.path,
      });
    });

    return await this.productImageRepository.save(productImages);
  }

  async remove(id: string) {
    const product = await this.productRepository.findOne({ where: { id } });
    if (!product) throw new NotFoundException(`Product not found`);

    await this.productRepository.remove(product);
    return { message: `Product has been deleted` };
  }
}
