import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Post,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { ApiOperation, ApiTags } from '@nestjs/swagger';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { FeedbackPaginationDto, RatingFeedbackDto } from './dto/rating-feedback.dto';
import { RatingFeedbackService } from './rating-feedback.service';
import { PermissionAction, UserRole } from 'src/enum';
import { query } from 'express';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';

@Controller('rating-feedback')
export class RatingFeedbackController {
  constructor(private readonly ratingFeedbackService: RatingFeedbackService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'))
  create(
    @Body() dto: RatingFeedbackDto, @CurrentUser() user: Account) {
    return this.ratingFeedbackService.rateProduct(dto, user.id);
  }

  @Get('admin')
@UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'rating_feedback'])
  async findAll(@Query() dto: FeedbackPaginationDto) {
    return this.ratingFeedbackService.findAll(dto);
  }
  
  @Get('user')
  @UseGuards(AuthGuard('jwt'))
  async getUserRatings(@CurrentUser() user: Account) {
    return this.ratingFeedbackService.findUserRatings(user.id);
  }

  @Put(':id')
 @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
   @CheckPermissions([PermissionAction.UPDATE, 'rating_feedback'])
  status(@Param('id') id: string, @Body() dto: BoolStatusDto) {
    return this.ratingFeedbackService.status(id, dto);
  }

  @Delete(':id')
@UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN,UserRole.STAFF)
  @CheckPermissions([PermissionAction.DELETE, 'rating_feedback'])
  remove(@Param('id') id: string) {
    return this.ratingFeedbackService.remove(id);
  }
}
