import { Injectable, NotAcceptableException, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { Brackets, Repository } from 'typeorm';
import {
  FeedbackPaginationDto,
  RatingFeedbackDto,
} from './dto/rating-feedback.dto';
import { RatingFeedback } from './entities/rating-feedback.entity';
import { Product } from 'src/product/entities/product.entity';



@Injectable()
export class RatingFeedbackService {
  constructor(
    @InjectRepository(RatingFeedback)
    private readonly repo: Repository<RatingFeedback>,
    @InjectRepository(Product) private readonly productRepo: Repository<Product>,


  ) { }
  async rateProduct(dto: RatingFeedbackDto, accountId: string) {
    const product = await this.productRepo.findOneBy({ id: dto.productId });
    if (!product) throw new NotFoundException('Product not found');

    const existingRating = await this.repo.findOne({
      where: { productId: product.id, accountId: accountId }
    });

    let rating;

    if (existingRating) {

      existingRating.rating = dto.rating;
      existingRating.feedback = dto.feedback;
      rating = await this.repo.save(existingRating);
    } else {
      rating = this.repo.create({
        productId: product.id,
        accountId: accountId,
        rating: dto.rating,
        feedback: dto.feedback,
      });
      await this.repo.save(rating);
    }
    const ratingStats = await this.averageRatingWithCount(product.id);
    await this.productRepo.update(product.id, {
      rating: parseFloat(ratingStats.average.toFixed(1)),
      reviewCount: ratingStats.count
    });

    return rating;
  }


  async findAll(dto: FeedbackPaginationDto) {
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('ratingFeedback')
      .leftJoinAndSelect('ratingFeedback.product', 'product')
      .leftJoinAndSelect('ratingFeedback.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'ratingFeedback.id',
        'ratingFeedback.feedback',
        'ratingFeedback.rating',
        'ratingFeedback.status',
        'product.id',
        'product.name',
        'ratingFeedback.createdAt',
      ])

    if (dto.status !== undefined) {
      query.andWhere('ratingFeedback.status = :status', { status: dto.status });
    }
    query.andWhere(
      new Brackets((qb) => {
        qb.where(
          'ratingFeedback.feedback LIKE :feedback ',
          {
            feedback: `%${keyword}%`,

          },
        );
      }),
    )
    query.take(dto.limit).skip(dto.offset)
    const [result, total] = await query.getManyAndCount();


    return { result, total };
  }

  async findUserRatings(accountId: string) {
    const [result, count] = await this.repo
      .createQueryBuilder('ratingFeedback')
      .leftJoinAndSelect('ratingFeedback.product', 'product')
      .leftJoinAndSelect('ratingFeedback.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'ratingFeedback.id',
        'ratingFeedback.feedback',
        'ratingFeedback.rating',
        'ratingFeedback.status',
        'ratingFeedback.createdAt',
        'product.id',
        'product.name',
        'account.id',
        'userDetail.id',
        'userDetail.name',
      ])
      .where('ratingFeedback.accountId = :accountId', { accountId })
      .orderBy('ratingFeedback.createdAt', 'DESC')
      .take(10)
      .getManyAndCount();
    return { result, count };
  }





  async averageRatingWithCount(productId: string) {
    try {
      const result = await this.repo
        .createQueryBuilder('ratingFeedback')
        .select('AVG(ratingFeedback.rating)', 'average')
        .addSelect('COUNT(ratingFeedback.id)', 'count')
        .where('ratingFeedback.productId = :productId', {
          productId,
        })
        .getRawOne();

     return {
  average: parseFloat(result.average) || 0,
  count: parseInt(result.count, 10) || 0,
};

    } catch (error) {
      console.error('Error fetching ratings:', error);
      return {
        average: 0,
        count: 0,
      };
    }
  }

  async status(id: string, dto: BoolStatusDto) {
    const feedBack = await this.repo.findOne({ where: { id } });
    if (!feedBack) {
      throw new NotAcceptableException('Feedback not found!');
    }
    const obj = Object.assign(feedBack, dto);
    await this.repo.save(obj);
    const ratingStats = await this.averageRatingWithCount(feedBack.productId);
    await this.productRepo.update(feedBack.productId, {
      rating: parseFloat(ratingStats.average.toFixed(1)),
      reviewCount: ratingStats.count
    });

    return obj;
  }

  async remove(id: string) {
    const feedBack = await this.repo.findOne({ where: { id } });
    if (!feedBack) {
      throw new NotAcceptableException('Feedback not found!');
    }

    const productId = feedBack.productId;
    await this.repo.remove(feedBack);
    const ratingStats = await this.averageRatingWithCount(productId);
    await this.productRepo.update(productId, {
      rating: parseFloat(ratingStats.average.toFixed(1)),
      reviewCount: ratingStats.count
    });

    return { message: 'Feedback removed successfully' };
  }
}
