import { Injectable, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Account } from 'src/account/entities/account.entity';
import { Brackets, Repository } from 'typeorm';
import { PaginationSDto, SaveProfileDto, UpdateUserDetailDto } from './dto/update-user-details';
import { UserDetail } from './entities/user-detail.entity';
import { DefaultStatus } from '../enum';
import * as fs from 'fs';
import * as path from 'path';

@Injectable()
export class UserDetailsService {
  constructor(
    @InjectRepository(UserDetail) private readonly repo: Repository<UserDetail>,
    @InjectRepository(Account)
    private readonly accountrepo: Repository<Account>,
  ) { }

  async profile(id: string) {
    const result = await this.repo
      .createQueryBuilder('userDetail')
      .leftJoinAndSelect('userDetail.account', 'account')

      .select([
        'userDetail.name',
        'userDetail.email',
        'userDetail.alternativePhoneNumber',
        'userDetail.profile',
        'userDetail.profileName',
        'userDetail.address',
        'account.phoneNumber',
      ])
      .where('userDetail.accountId = :id', { id: id })
      .getOne();
    if (!result) {
      throw new NotFoundException('User not found!');
    }
    return result;
  }

async findAll(dto: PaginationSDto) {
  const keyword = dto.keyword || '';
  
  const query = this.repo
    .createQueryBuilder('userDetail')
    .leftJoinAndSelect('userDetail.account', 'account')
    .where(
      new Brackets(qb => {
        qb.where('account.phoneNumber LIKE :keyword', { keyword: `%${keyword}%` })
          .orWhere('userDetail.name LIKE :keyword', { keyword: `%${keyword}%` });
      })
    );

  if (dto.status) {
    query.andWhere('account.status = :status', { status: dto.status });
  }

  const [result, total] = await query
    .skip(dto.offset || 0)
    .take(dto.limit || 10)
    .orderBy('userDetail.name', 'ASC')
    .getManyAndCount();

  return { result, total };
}


  async findOne(id: string) {
    const result = await this.repo.findOne({
      where: { accountId: id },
    });
    if (!result) {
      throw new NotFoundException('User not found!');
    }
    return result;
  }

  async update(dto: UpdateUserDetailDto, accountId: string) {
    const result = await this.repo.findOne({ where: { accountId: accountId } });
    if (!result) {
      throw new NotFoundException('User profile not found!');
    }
    const obj = Object.assign(result, dto);
    return await this.repo.save(obj);
  }

  async updateProfile(image: string | undefined, result: UserDetail, dto: UpdateUserDetailDto) {
    const obj = Object.assign(result, {
      ...dto,
      ...(image && {
        profile: process.env.RN_CDN_LINK + image,
        profileName: image,
      }),
    });

    return this.repo.save(obj);
  }

  async docoment(doc: string, result: UserDetail) {
    const obj = Object.assign(result, {
      documentUrl: process.env.RN_CDN_LINK + doc,
      documentPath: doc,
    });
    return this.repo.save(obj);
  }
}
