import PDFDocument from 'pdfkit';
import { Setting } from '../settings/entities/setting.entity';

export class PdfGeneratorUtil {
  static async generateInvoice(orderData: any, settings?: Setting): Promise<Buffer> {
    return new Promise((resolve) => {
      const doc = new PDFDocument({ size: 'A4', margin: 30, layout: 'portrait' });
      const buffers: Buffer[] = [];

      doc.on('data', buffers.push.bind(buffers));
      doc.on('end', () => resolve(Buffer.concat(buffers)));

      const leftX = 30;
      const rightX = 300;
      const pageWidth = 535;

      doc.fontSize(20).font('Helvetica-Bold').text('TAX INVOICE', { align: 'center' }).moveDown();

      doc.rect(leftX, doc.y, pageWidth, 120).stroke();
      const row1Y = doc.y;

      doc.fontSize(10).text('Sender:', leftX + 10, row1Y + 10).font('Helvetica-Bold');
      doc.text(settings?.companyName || 'RANUINFO PRIVATE LIMITED', leftX + 10, doc.y + 5);
      doc.font('Helvetica').text(settings?.companyAddress || 'NH-2 Delhi Road, Dankuni', leftX + 10, doc.y + 5);
      doc.text(settings?.companyCity || 'Hooghly - 712250, West Bengal, India', leftX + 10, doc.y + 5);
      doc.text(`Ph No: ${settings?.companyPhone || '9999999999'}`, leftX + 10, doc.y + 5);
      doc.text(`GSTIN: ${settings?.companyGstin || '19AADCI3821M1ZC'}`, leftX + 10, doc.y + 5);

      doc.moveTo(rightX, row1Y).lineTo(rightX, row1Y + 120).stroke();

      const createdAt = new Date(orderData.order.createdAt).toLocaleDateString();
      const orderId = orderData.order.id;
      const shortId = orderId.substring(0, 8).toUpperCase();

      doc.font('Helvetica-Bold');
      doc.text(`Invoice Code: INV/${shortId}`, rightX + 10, row1Y + 10);
      doc.text(`Invoice Date: ${createdAt}`, rightX + 10, doc.y + 5);
      doc.text(`Order No: ${orderId.slice(-6).toUpperCase()}`, rightX + 10, doc.y + 5);
      doc.text(`Order Date: ${createdAt}`, rightX + 10, doc.y + 5);
      doc.text(`Payment Mode: ${orderData.order.paymentMethod}`, rightX + 10, doc.y + 5);

      const row2Y = row1Y + 120;
      doc.fontSize(10).font('Helvetica-Bold');
      doc.text('Bill To:', leftX + 10, row2Y + 10);
      doc.text('Ship To:', rightX + 10, row2Y + 10);

      doc.font('Helvetica');
      const fullAddress = orderData.order.fullDeliveryAddress || 'No address available';
      const addressLines = fullAddress.split(', ');
      const lineHeight = 15;
      const boxHeight = addressLines.length * lineHeight + 30;

      doc.rect(leftX, row2Y, pageWidth, boxHeight).stroke();
      doc.moveTo(rightX, row2Y).lineTo(rightX, row2Y + boxHeight).stroke();

      let yBill = row2Y + 30;
      addressLines.forEach(line => {
        doc.text(line, leftX + 10, yBill);
        yBill += lineHeight;
      });

      let yShip = row2Y + 30;
      addressLines.forEach(line => {
        doc.text(line, rightX + 10, yShip);
        yShip += lineHeight;
      });

      const row3Y = row2Y + boxHeight;

     doc.rect(leftX, row3Y, pageWidth, 30).stroke();
doc.font('Helvetica-Bold').fontSize(10);
doc.text('SI', leftX + 10, row3Y + 10);
doc.text('Product Name', leftX + 40, row3Y + 10);
doc.text('Qty', leftX + 240, row3Y + 10, { width: 30, align: 'right' });
doc.text('MRP', leftX + 300, row3Y + 10, { width: 50, align: 'right' });
doc.text('Price', leftX + 360, row3Y + 10, { width: 50, align: 'right' });
doc.text('Total', leftX + 440, row3Y + 10, { width: 60, align: 'right' }); // adjusted

let currentY = row3Y + 30;
let totalQty = 0;
let totalAmount = 0;

if (orderData.order.orderItems.length === 0) {
  doc.rect(leftX, currentY, pageWidth, 30).stroke();
  doc.font('Helvetica').text('No items found.', leftX + 10, currentY + 10);
  currentY += 30;
} else {
  orderData.order.orderItems.forEach((item: any, index: number) => {
    const rowHeight = 30;
    doc.rect(leftX, currentY, pageWidth, rowHeight).stroke();

    const name = item.product?.name || 'Unnamed';
    const mrp = Number(item.product?.price || 0);
    const price = Number(item.price);
    const qty = Number(item.quantity);
    const totalprice = Number(item.totalPrice);

    totalQty += qty;
    totalAmount += totalprice;

    doc.font('Helvetica').fontSize(10);
    doc.text(`${index + 1}`, leftX + 10, currentY + 10);
    doc.text(`${name}`, leftX + 40, currentY + 10, { width: 190 });
    doc.text(qty.toString(), leftX + 240, currentY + 10, { width: 30, align: 'right' });
    doc.text(`${Math.round(mrp).toLocaleString('en-IN')}`, leftX + 300, currentY + 10, { width: 50, align: 'right' });
    doc.text(`${Math.round(price).toLocaleString('en-IN')}`, leftX + 360, currentY + 10, { width: 50, align: 'right' });
    doc.text(`${Math.round(totalprice).toLocaleString('en-IN')}`, leftX + 440, currentY + 10, { width: 60, align: 'right' });

    currentY += rowHeight;
  }); 

}
      const gstAmount = Number(orderData.order.gstAmount || 0);
      const deliveryCharge = Number(orderData.order.deliveryCharge || 0);
      const grandTotal = totalAmount + gstAmount + deliveryCharge;

      // Subtotal Row
      doc.rect(leftX, currentY, pageWidth, 30).stroke();
      doc.font('Helvetica-Bold');
      doc.text('Subtotal:', leftX + 360, currentY + 10);
      doc.text(`${Math.round(totalAmount)}`, leftX + 470, currentY + 10, { width: 40, align: 'right' });
      currentY += 30;

      // GST Row
      doc.rect(leftX, currentY, pageWidth, 30).stroke();
      doc.font('Helvetica-Bold');
      doc.text('GST:', leftX + 360, currentY + 10);
      doc.text(`${Math.round(gstAmount)}`, leftX + 470, currentY + 10, { width: 40, align: 'right' });
      currentY += 30;

      // Delivery Charge Row
      doc.rect(leftX, currentY, pageWidth, 30).stroke();
      doc.font('Helvetica-Bold');
      doc.text('Delivery Charge:', leftX + 360, currentY + 10);
      doc.text(`${deliveryCharge}`, leftX + 470, currentY + 10, { width: 40, align: 'right' });
      currentY += 30;

      // Grand Total Row
      doc.rect(leftX, currentY, pageWidth, 30).stroke();
      doc.font('Helvetica-Bold').fontSize(11);
      doc.text('Grand Total:', leftX + 360, currentY + 10);
      doc.text(`${Math.round(grandTotal)}`, leftX + 470, currentY + 10, { width: 40, align: 'right' });
      currentY += 30;

      // Amount in Words
      doc.rect(leftX, currentY, pageWidth, 40).stroke();
      doc.font('Helvetica');
      doc.text(`Amount Chargeable (in words):`, leftX + 10, currentY + 10);
      doc.font('Helvetica-Bold').text(`${Math.round(grandTotal).toString().toUpperCase()} RUPEES ONLY`, leftX + 10, currentY + 25);
      currentY += 40;

      // Declaration
      doc.fontSize(9).font('Helvetica');
      const declarationText = settings?.invoiceDeclaration || '1. All claims for shortages or damages must be reported on the day of delivery.';
      const textOptions = { width: pageWidth - 20, align: 'left' as const };
      const textHeight = doc.heightOfString(declarationText, textOptions);
      const declarationHeight = Math.max(60, textHeight + 40);

      doc.rect(leftX, currentY, pageWidth, declarationHeight).stroke();
      doc.text('Declaration:', leftX + 10, currentY + 10);
      doc.text(declarationText, leftX + 10, currentY + 25, textOptions);
      currentY += declarationHeight;

      // Signature
      doc.rect(leftX, currentY, pageWidth, 100).stroke();
      doc.text(`For ${settings?.companyName || 'RANUINFOTECH PRIVATE LIMITED'}`, leftX + 50, currentY + 30);
      doc.text('Authorised Signatory', leftX + 50, currentY + 60);

      // Footer
      const footerText = `This is a computer-generated invoice`;
      doc.fontSize(9).text(footerText, leftX, currentY + 80, { align: 'center', width: pageWidth });

      doc.end();
    });
  }
}