import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, Query } from '@nestjs/common';
import { WishlistService } from './wishlist.service';
import { CreateWishlistDto } from './dto/wishlist.dto';
import { UserRole } from 'src/enum';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { AuthGuard } from '@nestjs/passport';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { CommonPaginationDto } from '../common/dto/common-pagination.dto';

@Controller('wishlist')
export class WishlistController {
  constructor(private readonly wishlistService: WishlistService) { }

  @Post()
  @UseGuards(AuthGuard('jwt'))
  create(@Query() dto: CreateWishlistDto,
    @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.wishlistService.create(dto);
  }

  @Get('user')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER, )
  async getUserWishlist(@Query() dto: CommonPaginationDto, @CurrentUser() user: Account) {
    return this.wishlistService.getProductsForUser(user.id, dto);
  }

  @Get('retailer')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.RETAILER, )
  async getRetailerWishlist(@Query() dto: CommonPaginationDto, @CurrentUser() user: Account) {
    return this.wishlistService.getProductsForRetailer(user.id, dto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'))
  remove(@Param('id') id: string,
    @CurrentUser() user: Account) {
    const accountId = user.id;
    return this.wishlistService.removes(id, accountId);
  }
}
