import { ConflictException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateWishlistDto } from './dto/wishlist.dto';
import { Wishlist } from './entities/wishlist.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { UserRole } from '../enum';
import { CommonPaginationDto } from '../common/dto/common-pagination.dto';


@Injectable()
export class WishlistService {

  constructor(

    @InjectRepository(Wishlist) private readonly wishrepo: Repository<Wishlist>,
  ) { }
  async create(Dto: CreateWishlistDto) {
    const wishlist = await this.wishrepo.findOne({
      where: { productId: Dto.productId, accountId: Dto.accountId },
    });
    if (wishlist) {
      throw new ConflictException('Product already exists in wishlist!');

    }
    const obj = Object.create(Dto);
    await this.wishrepo.save(obj);
    return { message: 'Product added to wishlist successfully' };

  }

  async getProductsForUser(accountId: string, dto: CommonPaginationDto) {
    const queryBuilder = this.wishrepo
      .createQueryBuilder('wishlist')
      .leftJoinAndSelect('wishlist.product', 'product')
      .where('wishlist.accountId = :accountId', { accountId })
      .select([
        'wishlist.id',
        'wishlist.createdAt',
        'wishlist.updatedAt',
        'product.id',
        'product.name',
        'product.price',
        'product.imageUrl',
        'product.description',
        'product.rating',
        'product.userPrice',
      ])
      .orderBy('wishlist.createdAt', 'DESC');
    const [result, total] = await queryBuilder
      .limit(dto.limit)
      .offset(dto.offset)
      .getManyAndCount();

    if (!result || result.length === 0) {
      throw new NotFoundException('Products not found in wishlist!');
    }

    return { result, total };
  }
  async getProductsForRetailer(accountId: string, dto: CommonPaginationDto) {
    const queryBuilder = this.wishrepo
      .createQueryBuilder('wishlist')
      .leftJoinAndSelect('wishlist.product', 'product')
      .where('wishlist.accountId = :accountId', { accountId })
      .select([
        'wishlist.id',
        'wishlist.createdAt',
        'wishlist.updatedAt',
        'product.id',
        'product.price',
        'product.name',
        'product.imageUrl',
        'product.description',
        'product.rating',
        'product.retailerPrice',
      ])
      .orderBy('wishlist.createdAt', 'DESC');

    const [result, total] = await queryBuilder
      .limit(dto.limit)
      .offset(dto.offset)
      .getManyAndCount();

    if (!result || result.length === 0) {
      throw new NotFoundException('Products not found in wishlist!');
    }

    return { result, total };
  }


  async removes(id: string, accountId: string) {
    const product = await this.wishrepo.findOne({ where: { productId: id, accountId: accountId } });
    if (!product) {
      throw new NotFoundException('Product not found');
    }
    await this.wishrepo.remove(product);
    return ({ message: 'Product  remove successfully' });
  }


}
